<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'WooCommerce' ) ) {
	return;
}

use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Image_Size;
use Elementor\Repeater;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Background;


/**
 * Elementor tabs widget.
 *
 * Elementor widget that displays vertical or horizontal tabs with different
 * pieces of content.
 *
 * @since 1.0.0
 */
class TemplateMelaCore_Elementor_Products_Tabs extends Elementor\Widget_Base {

	public function get_categories() {
		return array( 'tmcore-addons' );
	}

	/**
	 * Get widget name.
	 *
	 * Retrieve tabs widget name.
	 *
	 * @since  1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'tmcore-products-tabs';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve tabs widget title.
	 *
	 * @since  1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'TM Products Tabs', 'templatemela-core' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve tabs widget icon.
	 *
	 * @since  1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-tabs';
	}


	public function get_script_depends() {
		return [ 'tmcore-elementor-product-tab', 'base-splide' ];
	}

	/**
	 * Register tabs widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since  1.0.0
	 * @access protected
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'section_tabs',
			[
				'label' => __( 'Tabs', 'templatemela-core' ),
			]
		);

		$repeater = new Repeater();

		$repeater->add_control(
			'tab_title',
			[
				'label'       => __( 'Tab Title', 'templatemela-core' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => __( '#Product Tab', 'templatemela-core' ),
				'placeholder' => __( 'Product Tab Title', 'templatemela-core' ),
			]
		);

		$repeater->add_control(
            'tab_image',
            [
                'label'      => __('Choose Image', 'templatemela-core'),
                'type'       => Controls_Manager::MEDIA,
                'show_label' => false,
            ]
        );

		$repeater->add_control(
			'limit',
			[
				'label'   => __( 'Posts Per Page', 'templatemela-core' ),
				'type'    => Controls_Manager::NUMBER,
				'default' => 6,
			]
		);

		$repeater->add_control(
			'advanced',
			[
				'label' => __( 'Advanced', 'templatemela-core' ),
				'type'  => Controls_Manager::HEADING,
			]
		);

		$repeater->add_control(
			'orderby',
			[
				'label'   => __( 'Order By', 'templatemela-core' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'date',
				'options' => [
					'date'       => __( 'Date', 'templatemela-core' ),
					'id'         => __( 'Post ID', 'templatemela-core' ),
					'menu_order' => __( 'Menu Order', 'templatemela-core' ),
					'popularity' => __( 'Number of purchases', 'templatemela-core' ),
					'rating'     => __( 'Average Product Rating', 'templatemela-core' ),
					'title'      => __( 'Product Title', 'templatemela-core' ),
					'rand'       => __( 'Random', 'templatemela-core' ),
				],
			]
		);

		$repeater->add_control(
			'order',
			[
				'label'   => __( 'Order', 'templatemela-core' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'desc',
				'options' => [
					'asc'  => __( 'ASC', 'templatemela-core' ),
					'desc' => __( 'DESC', 'templatemela-core' ),
				],
			]
		);

		$repeater->add_control(
			'categories',
			[
				'label'    => __( 'Categories', 'templatemela-core' ),
				'type'     => Controls_Manager::SELECT2,
				'label_block' => true,
				'options'  => $this->get_product_categories(),
				'multiple' => true,
			]
		);

		$repeater->add_control(
			'cat_operator',
			[
				'label'     => __( 'Category Operator', 'templatemela-core' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => 'IN',
				'options'   => [
					'AND'    => __( 'AND', 'templatemela-core' ),
					'IN'     => __( 'IN', 'templatemela-core' ),
					'NOT IN' => __( 'NOT IN', 'templatemela-core' ),
				],
				'condition' => [
					'categories!' => ''
				],
			]
		);

		$repeater->add_control(
			'tag',
			[
				'label'    => __( 'Tags', 'templatemela-core' ),
				'type'     => Controls_Manager::SELECT2,
				'label_block' => true,
				'options'  => $this->get_product_tags(),
				'multiple' => true,
			]
		);

		$repeater->add_control(
			'tag_operator',
			[
				'label'     => __( 'Tag Operator', 'templatemela-core' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => 'IN',
				'options'   => [
					'AND'    => __( 'AND', 'templatemela-core' ),
					'IN'     => __( 'IN', 'templatemela-core' ),
					'NOT IN' => __( 'NOT IN', 'templatemela-core' ),
				],
				'condition' => [
					'tag!' => ''
				],
			]
		);

		$repeater->add_control(
			'product_type',
			[
				'label'   => __( 'Product Type', 'templatemela-core' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'newest',
				'options' => [
					'newest'       => __( 'Newest Products', 'templatemela-core' ),
					'on_sale'      => __( 'On Sale Products', 'templatemela-core' ),
					'best_selling' => __( 'Best Selling', 'templatemela-core' ),
					'top_rated'    => __( 'Top Rated', 'templatemela-core' ),
					'featured'     => __( 'Featured Product', 'templatemela-core' ),
				],
			]
		);

		$repeater->add_control(
			'product_layout',
			[
				'label'   => __( 'Product Layout', 'templatemela-core' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'grid',
				'options' => [
					'grid' => __( 'Grid', 'templatemela-core' ),
					'list' => __( 'List', 'templatemela-core' ),
				],
			]
		);

		$repeater->add_control(
			'list_layout',
			[
				'label'     => __( 'List Layout', 'templatemela-core' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => '1',
				'options'   => [
					'1' => __( 'Style 1', 'templatemela-core' ),
					'2' => __( 'Style 2', 'templatemela-core' ),
					'3' => __( 'Style 3', 'templatemela-core' ),
					'4' => __( 'Style 4', 'templatemela-core' ),
					'5' => __( 'Style 5', 'templatemela-core' ),
				],
				'condition' => [
					'product_layout' => 'list'
				]
			]
		);

		$repeater->add_responsive_control(
			'list_image_size',
			[
				'label' => esc_html__( 'Image Width', 'templatemela-core' ) . ' (%)',
				'type'       => Controls_Manager::SLIDER,
				'default' => [
					'size' => 30,
					'unit' => '%',
				],
				'laptop_default' => [
					'unit' => '%',
				],
				'tablet_default' => [
					'unit' => '%',
				],
				'mobile_default' => [
					'unit' => '%',
				],
				'size_units' => [ 'px', '%', 'em', 'rem', 'vw', 'custom' ],
				'range' => [
					'%' => [
						'min' => 5,
						'max' => 100,
					],
				],
				'selectors'  => [
					'{{WRAPPER}} .products-list img'      =>  'width: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'product_layout' => 'list'
				]
			]
        );

        $repeater->add_control(
            'hide_except',
            [
                'label'      => esc_html__('Hide Except', 'templatemela-core'),
                'type'       => Controls_Manager::SWITCHER,
                'conditions' => [
                    'relation' => 'and',
                    'terms'    => [
                        [
                            'name'     => 'product_layout',
                            'operator' => '==',
                            'value'    => 'list',
                        ],
                        [
							'relation' => 'or',
							'terms' => [
								[
                                    'name'     => 'list_layout',
                                    'operator' => '==',
                                    'value'    => '3',
                                ],
								[
                                    'name'     => 'list_layout',
                                    'operator' => '==',
                                    'value'    => '5',
                                ],
							],
						],
                    ],
                ]
            ]
        );

        $repeater->add_control(
            'hide_category',
            [
                'label'      => esc_html__('Hide Category', 'templatemela-core'),
                'type'       => Controls_Manager::SWITCHER,
                'conditions' => [
                    'relation' => 'and',
                    'terms'    => [
                        [
                            'name'     => 'product_layout',
                            'operator' => '==',
                            'value'    => 'list',
                        ],
                        [
							'relation' => 'or',
							'terms' => [
								[
                                    'name'     => 'list_layout',
                                    'operator' => '==',
                                    'value'    => '2',
                                ],
                                [
                                    'name'     => 'list_layout',
                                    'operator' => '==',
                                    'value'    => '3',
                                ],
                                [
                                    'name'     => 'list_layout',
                                    'operator' => '==',
                                    'value'    => '5',
                                ],
							],
						],
                    ],
                ]
            ]
        );

        $repeater->add_control(
            'hide_button',
            [
                'label'      => esc_html__('Hide Button', 'templatemela-core'),
                'type'       => Controls_Manager::SWITCHER,
                'conditions' => [
                    'relation' => 'and',
                    'terms'    => [
                        [
                            'name'     => 'product_layout',
                            'operator' => '==',
                            'value'    => 'list',
                        ],
                        [
							'relation' => 'or',
							'terms' => [
								[
                                    'name'     => 'list_layout',
                                    'operator' => '==',
                                    'value'    => '2',
                                ],
                                [
                                    'name'     => 'list_layout',
                                    'operator' => '==',
                                    'value'    => '4',
                                ],
							],
						],
                    ],
                ]
            ]
        );

        $repeater->add_control(
            'hide_rating',
            [
                'label'      => esc_html__('Hide Rating', 'templatemela-core'),
                'type'       => Controls_Manager::SWITCHER,
                'conditions' => [
                    'relation' => 'and',
                    'terms'    => [
                        [
                            'name'     => 'product_layout',
                            'operator' => '==',
                            'value'    => 'list',
                        ],
                        [
							'relation' => 'or',
							'terms' => [
								[
                                    'name'     => 'list_layout',
                                    'operator' => '==',
                                    'value'    => '4',
                                ],
                                [
                                    'name'     => 'list_layout',
                                    'operator' => '==',
                                    'value'    => '5',
                                ],
							],
						],
                    ],
                ]
            ]
        );

		$repeater->add_control(
            'show_sale_countdown',
            [
                'label' => esc_html__('Show Sale Countdown?', 'templatemela-core'),
                'type'  => Controls_Manager::SWITCHER,
                /* 'condition' => [
                    'product_type' => 'on_sale'
                ] */
            ]
        );

		$this->add_control(
			'tabs',
			[
				'label'       => '',
				'type'        => Controls_Manager::REPEATER,
				'fields'      => $repeater->get_controls(),
				'default'     => [
					[
						'tab_title' => __( '#Product Tab 1', 'templatemela-core' ),
					],
					[
						'tab_title' => __( '#Product Tab 2', 'templatemela-core' ),
					]
				],
				'title_field' => '{{{ tab_title }}}',
			]
		);

		$this->add_group_control(
            Elementor\Group_Control_Image_Size::get_type(),
            [
                'name'      => 'tab_image',
                // Usage: `{name}_size` and `{name}_custom_dimension`, in this case `tab_image_size` and `tab_image_custom_dimension`.
                'default'   => 'full',
                'separator' => 'none',
            ]
        );

		$this->add_responsive_control(
			'column',
			[
				'label'          => __( 'columns', 'templatemela-core' ),
				'type'           => \Elementor\Controls_Manager::SELECT,
				'default'        => 6,
				'laptop_default' => 4,
				'tablet_default' => 3,
				'mobile_default' => 2,
				'options'        => [1 => 1, 2 => 2, 3 => 3, 4 => 4, 5 => 5, 6 => 6, 7 => 7, 8 => 8],
			]
		);

		$this->add_control(
			'type',
			[
				'label'        => esc_html__( 'Type', 'templatemela-core' ),
				'type'         => Controls_Manager::SELECT,
				'default'      => 'horizontal',
				'options'      => [
					'horizontal' => esc_html__( 'Horizontal', 'templatemela-core' ),
					'vertical'   => esc_html__( 'Vertical', 'templatemela-core' ),
				],
				'prefix_class' => 'elementor-tabs-view-',
				'separator'    => 'before',
			]
		);

		// $this->add_control(
        //     'show_sale_countdown',
        //     [
        //         'label' => esc_html__('Show Sale Countdown?', 'templatemela-core'),
        //         'type'  => Controls_Manager::SWITCHER,
        //         /* 'condition' => [
        //             'product_type' => 'on_sale'
        //         ] */
        //     ]
        // );

		$this->end_controls_section();

		$this->start_controls_section(
			'section_product',
			[
				'label' => __( 'Product', 'templatemela-core' ),
				'condition'   => [
                    'enable_carousel!' => 'yes'
                ]
			]
		);

		$this->add_responsive_control(
			'product_gutter',
			[
				'label'      => __( 'Grid Gutter', 'templatemela-core' ),
				'type'       => Controls_Manager::SLIDER,
				'default'    =>  [
					'size' => 20,
					'unit' => 'px',
				],
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'size_units' => [ 'px', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} ul.products.grid-cols li.product'      => 'margin-left: calc({{SIZE}}{{UNIT}} / 2); margin-right: calc({{SIZE}}{{UNIT}} / 2); margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} ul.products.grid-cols li.product-item' => 'margin-left: calc({{SIZE}}{{UNIT}} / 2); margin-right: calc({{SIZE}}{{UNIT}} / 2); margin-bottom: calc({{SIZE}}{{UNIT}} - 1px);',
					'{{WRAPPER}} .products-list ul.products.grid-cols > li'            => 'margin-left: calc({{SIZE}}{{UNIT}} / 2); margin-right: calc({{SIZE}}{{UNIT}} / 2); margin-bottom: calc({{SIZE}}{{UNIT}} - 1px);',
					'{{WRAPPER}} ul.products.grid-cols'                 => 'margin-left: calc({{SIZE}}{{UNIT}} / -2); margin-right: calc({{SIZE}}{{UNIT}} / -2);',
				]
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_tab_header_style',
			[
				'label' => __( 'Header', 'templatemela-core' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'tab_header_padding',
			[
				'label'      => __( 'Padding', 'templatemela-core' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .elementor-tabs-wrapper' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
            'tab_header_margin',
            [
                'label'      => __('Margin', 'templatemela-core'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-tabs-wrapper' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

		$this->add_control(
			'background_tab_header',
			[
				'label'     => __( 'Background Color', 'templatemela-core' ),
				'type'      => Controls_Manager::COLOR,
				'separator'   => 'before',
				'selectors' => [
					'{{WRAPPER}} .elementor-tabs-wrapper' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name'        => 'tab_header_border',
                'placeholder' => '1px',
                'default'     => '1px',
                'selector'    => '{{WRAPPER}} .elementor-tabs-wrapper',
            ]
        );

		$this->add_control(
            'tab_header_radius',
            [
                'label'      => __('Border Radius', 'templatemela-core'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-tabs-wrapper' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

		$this->add_responsive_control(
			'align_items',
			[
				'label'        => __( 'Align', 'templatemela-core' ),
				'type'         => Controls_Manager::CHOOSE,
				'label_block'  => false,
				'separator'   => 'before',
				'options'      => [
					'left'   => [
						'title' => __( 'Left', 'templatemela-core' ),
						'icon'  => 'eicon-h-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'templatemela-core' ),
						'icon'  => 'eicon-h-align-center',
					],
					'right'  => [
						'title' => __( 'Right', 'templatemela-core' ),
						'icon'  => 'eicon-h-align-right',
					],
				],
				'default'      => '',
				'prefix_class' => 'elementor-tabs-h-align-',
				'selectors'    => [
					'{{WRAPPER}} .elementor-tabs-wrapper .elementor-tabs-items' => 'justify-content: {{VALUE}};',
				],
			]
		);
		
		$this->add_responsive_control(
            'header_margin',
            [
                'label'      => __('Margin Between Item', 'templatemela-core'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-tabs-wrapper .elementor-tab-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

		$this->add_responsive_control(
			'tab_header_width',
			[
				'label' => esc_html__( 'Width', 'templatemela-core' ) . ' (%)',
				'type' => Controls_Manager::SLIDER,
                'condition' => [
                    'type' => [
                        'vertical',
                    ],
				],
				'default' => [
					'size' => 30,
					'unit' => '%',
				],
				'laptop_default' => [
					'unit' => '%',
				],
				'tablet_default' => [
					'unit' => '%',
				],
				'mobile_default' => [
					'unit' => '%',
				],
				'size_units' => [ 'px', '%', 'em', 'rem', 'vw', 'custom' ],
				'range' => [
					'%' => [
						'min' => 5,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .elementor-tabs-wrapper' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_title_style',
			[
				'label' => __( 'Title', 'templatemela-core' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'show_divider',
			[
				'label'     => __( 'Show Divider', 'templatemela-core' ),
				'type'      => Controls_Manager::SWITCHER,
				'selectors' => [
					'{{WRAPPER}} .elementor-tab-title:before' => 'content: "";'
				]
			]
		);

		$this->add_control(
			'show_divider_color',
			[
				'label'     => __( 'Divider Color', 'templatemela-core' ),
				'type'      => Controls_Manager::COLOR,
				'condition' => [
                    'show_divider' => 'yes'
                ],
				'selectors' => [
					'{{WRAPPER}} .elementor-tab-title:before' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'tab_typography',
				'selector' => '{{WRAPPER}} .elementor-tab-title',
			]
		);

		$this->add_responsive_control(
			'tab_title_spacing',
			[
				'label'      => __( 'Spacing', 'templatemela-core' ),
				'type'       => Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 300,
					],
				],
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .elementor-tab-title' => 'padding-right: calc( {{SIZE}}{{UNIT}}/2 ); padding-left: calc( {{SIZE}}{{UNIT}}/2 );',
				],
			]
		);

		$this->add_responsive_control(
			'tab_title_vertical_padding',
			[
				'label'      => __( 'Vertical Padding', 'templatemela-core' ),
				'type'       => Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 10,
					],
				],
				'size_units' => [ 'px', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} .elementor-tab-title' => 'padding-top: {{SIZE}}{{UNIT}}; padding-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->start_controls_tabs( 'tabs_title_style' );

		$this->start_controls_tab(
			'tab_title_normal',
			[
				'label' => __( 'Normal', 'templatemela-core' ),
			]
		);

		$this->add_control(
			'title_color',
			[
				'label'     => __( 'Color', 'templatemela-core' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .elementor-tab-title' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'title_background_color',
			[
				'label'     => __( 'Background Color', 'templatemela-core' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .elementor-tab-title' => 'background-color: {{VALUE}};'
				],
			]
		);


		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_title_hover',
			[
				'label' => __( 'Hover', 'templatemela-core' ),
			]
		);

		$this->add_control(
			'title_hover_color',
			[
				'label'     => __( 'Color', 'templatemela-core' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .elementor-tab-title:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'title_background_hover_color',
			[
				'label'     => __( 'Background Color', 'templatemela-core' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .elementor-tab-title:hover' => 'background-color: {{VALUE}}'
				],
			]
		);

		$this->add_control(
			'title_hover_border_color',
			[
				'label'     => __( 'Border Color', 'templatemela-core' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .elementor-tab-title:hover' => 'border-color: {{VALUE}}'
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_title_active',
			[
				'label' => __( 'Active', 'templatemela-core' ),
			]
		);

		$this->add_control(
			'title_active_color',
			[
				'label'     => __( 'Color', 'templatemela-core' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .elementor-tab-title.elementor-active'        => 'color: {{VALUE}}',
					'{{WRAPPER}} .elementor-tab-title.elementor-active:before' => 'background:',
				],
			]
		);

		$this->add_control(
			'title_background_active_color',
			[
				'label'     => __( 'Background Color', 'templatemela-core' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .elementor-tab-title.elementor-active' => 'background-color: {{VALUE}} !important;',
				],
			]
		);

		$this->add_control(
			'title_active_border_color',
			[
				'label'     => __( 'Border Color', 'templatemela-core' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .elementor-tab-title.elementor-active' => 'border-color: {{VALUE}}!important;'
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'        => 'border_tabs_title',
				'placeholder' => '1px',
				'default'     => '1px',
				'selector'    => '{{WRAPPER}} .elementor-tab-title',
				'separator'   => 'before',
			]
		);

        $this->add_control(
            'border_tabs_title_radius',
            [
                'label'      => __( 'Border radius', 'templatemela-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-tab-title' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

		$this->end_controls_section();

		$this->add_control_carousel();

	}

	/**
	 * Render tabs widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since  1.0.0
	 * @access protected
	 */
	protected function render() {
		$tabs    = $this->get_settings_for_display( 'tabs' );
		$settings = $this->get_settings_for_display();

		$id_int = substr( $this->get_id_int(), 0, 3 );

		$this->add_render_attribute( 'data-carousel', 'class', 'elementor-tabs-content-wrapper' );

		if ( $settings['enable_carousel'] ) {

			$carousel_settings = $this->get_carousel_settings( $settings );
			$this->add_render_attribute( 'data-carousel', 'data-settings', wp_json_encode( $carousel_settings ) );
		}

		?>
        <div class="elementor-tabs" role="tablist">
            <div class="elementor-tabs-wrapper">
				<div class="elementor-tabs-items">
				<?php
				foreach ( $tabs as $index => $item ) :
					$tab_count = $index + 1;
					$class_item = 'elementor-repeater-item-' . $item['_id'];
					$class = ( $index == 0 ) ? 'elementor-active' : '';

					$tab_title_setting_key = $this->get_repeater_setting_key( 'tab_title', 'tabs', $index );

					$this->add_render_attribute( $tab_title_setting_key, [
						'id'            => 'elementor-tab-title-' . $id_int . $tab_count,
						'class'         => [
							'elementor-tab-title',
							'elementor-tab-desktop-title',
							$class,
							$class_item
						],
						'data-tab'      => $tab_count,
						'tabindex'      => $id_int . $tab_count,
						'role'          => 'tab',
						'aria-controls' => 'elementor-tab-content-' . $id_int . $tab_count,
					] );
					 
					?>
					
                    <div <?php echo tmcore_elementor_get_render_attribute_string( $tab_title_setting_key, $this ); // WPCS: XSS ok. ?>>
					<?php 
						$this->render_image($settings, $item);
						echo esc_html( $item['tab_title'] ); 
					?>
					</div>
				<?php endforeach; ?>
            </div>
			</div>
            <div <?php echo tmcore_elementor_get_render_attribute_string( 'data-carousel', $this ); // WPCS: XSS ok.
			?>>
				<?php
				foreach ( $tabs as $index => $item ) :
					$tab_count = $index + 1;
					$class_item = 'elementor-repeater-item-' . $item['_id'];
					$class_content = ( $index == 0 ) ? 'elementor-active' : '';

					$tab_content_setting_key = $this->get_repeater_setting_key( 'tab_content', 'tabs', $index );


					if ( $index == 0 ) {
						$this->add_render_attribute( $tab_content_setting_key, [
							'id'              => 'elementor-tab-content-' . $id_int . $tab_count,
							'class'           => [
								'elementor-tab-content',
								'elementor-clearfix',
								$class_content,
								$class_item
							],
							'data-tab'        => $tab_count,
							'role'            => 'tabpanel',
							'aria-labelledby' => 'elementor-tab-title-' . $id_int . $tab_count,
						] );
					} else {
					$this->add_render_attribute( $tab_content_setting_key, [
						'id'              => 'elementor-tab-content-' . $id_int . $tab_count,
						'class'           => [
							'elementor-tab-content',
							'elementor-clearfix',
							$class_content,
							$class_item
						],
						'data-tab'        => $tab_count,
						'role'            => 'tabpanel',
						'aria-labelledby' => 'elementor-tab-title-' . $id_int . $tab_count,
							'hidden'          => '',
					] );
					}
					

					$this->add_inline_editing_attributes( $tab_content_setting_key, 'advanced' );
					?>

                    <div <?php echo tmcore_elementor_get_render_attribute_string( $tab_content_setting_key, $this ); // WPCS: XSS ok. ?>>
					<?php 
						$this->woocommerce_default( $item, $settings ); 
					?>
                    </div>
				<?php endforeach; ?>
            </div>
        </div>
		<?php
	}

	private function render_image($settings, $item) {
        if (!empty($item['tab_image']['url'])) :
            ?>
            <div class="elementor-tab-image">
                <?php
                $item['tab_image_size']             = $settings['tab_image_size'];
				$item['tab_image_custom_dimension'] = $settings['tab_image_custom_dimension'];
                echo Group_Control_Image_Size::get_attachment_image_html($item, 'tab_image');
                ?>
            </div>
        <?php
        endif;
    }

	private function woocommerce_default( $settings, $global_setting ) {
		$type = 'products';

		$class = '';

		if ( $global_setting['enable_carousel'] ) {

			$atts['product_layout'] = 'carousel';
			$atts                   = [
				'limit'             => $settings['limit'],
				'orderby'           => $settings['orderby'],
				'order'             => $settings['order'],
				'carousel_settings' => '',
				'columns'           => $global_setting['column'],
				'product_layout'    => 'carousel'
			];

			if ( $settings['product_layout'] == 'list' ) {
				$atts['product_layout'] = 'list-carousel';
			}
		} else {
			$atts = [
				'limit'             => $settings['limit'],
				'orderby'           => $settings['orderby'],
				'order'             => $settings['order'],
				'carousel_settings' => '',
				'columns'           => $global_setting['column'],
				'product_layout'    => $settings['product_layout']
			];

			if ( ! empty( $global_setting['column_laptop'] ) ) {
				$class .= ' columns-laptop-' . $global_setting['column_laptop'];
			}

			if ( ! empty( $global_setting['column_tablet'] ) ) {
				$class .= ' columns-tablet-' . $global_setting['column_tablet'];
			}

			if ( ! empty( $global_setting['column_mobile'] ) ) {
				$class .= ' columns-mobile-' . $global_setting['column_mobile'];
			}
		}

		if ( $settings['product_layout'] == 'list' ) {
			
			$class               .= ' products-list';
			$class               .= ' products-list-' . $settings['list_layout'];
			$class               .= ' woocommerce-product-list';

			if (!empty($settings['list_layout']) && $settings['list_layout'] == '2') {
                $atts['show_category'] = true;
                $atts['show_button']   = true;
            }

            if (!empty($settings['list_layout']) && $settings['list_layout'] == '3') {
                $atts['show_category'] = true;
                $atts['show_except']   = true;
            }

            if (!empty($settings['list_layout']) && $settings['list_layout'] == '4') {
                $atts['show_rating']    = true;
                $atts['show_button']   = true;
            }

            if (!empty($settings['list_layout']) && $settings['list_layout'] == '5') {
                $atts['show_category']  = true;
                $atts['show_except']    = true;
                //$atts['show_countdown'] = true;
                $atts['show_rating']    = true;
            }

            if ($settings['hide_except'] == 'yes') {
                $atts['show_except'] = '';
            }

            if ($settings['hide_category'] == 'yes') {
                $atts['show_category'] = '';
            }

            if ($settings['hide_button'] == 'yes') {
                $atts['show_button'] = '';
            }

            if ($settings['hide_rating'] == 'yes') {
                $atts['show_rating'] = '';
            }

		}
		if (!empty($settings['show_sale_countdown']) && $settings['show_sale_countdown'] == 'yes') {
            $atts['show_countdown'] = true;   
        }

		$atts = $this->get_product_type( $atts, $settings['product_type'] );
		if ( isset( $atts['on_sale'] ) && wc_string_to_bool( $atts['on_sale'] ) ) {
			$type = 'sale_products';
		} elseif ( isset( $atts['best_selling'] ) && wc_string_to_bool( $atts['best_selling'] ) ) {
			$type = 'best_selling_products';
		} elseif ( isset( $atts['top_rated'] ) && wc_string_to_bool( $atts['top_rated'] ) ) {
			$type = 'top_rated_products';
		}

		if ( ! empty( $settings['categories'] ) ) {
			$atts['category']     = implode( ',', $settings['categories'] );
			$atts['cat_operator'] = $settings['cat_operator'];
		}

		if ( ! empty( $settings['tag'] ) ) {
			$atts['tag']          = implode( ',', $settings['tag'] );
			$atts['tag_operator'] = $settings['tag_operator'];
		}

		$atts['class'] = $class;

		$atts['cache'] = false; // disable transients for cache

		echo ( new WC_Shortcode_Products( $atts, $type ) )->get_content(); // WPCS: XSS ok.
	}

	protected function get_product_type( $atts, $product_type ) {
		switch ( $product_type ) {
			case 'featured':
				$atts['visibility'] = "featured";
				break;

			case 'on_sale':
				$atts['on_sale'] = true;
				break;

			case 'best_selling':
				$atts['best_selling'] = true;
				break;

			case 'top_rated':
				$atts['top_rated'] = true;
				break;

			default:
				break;
		}

		return $atts;
	}

	protected function get_product_tags() {
		$tags    = get_terms( array(
				'taxonomy'   => 'product_tag',
				'hide_empty' => false,
			)
		);
		$results = array();
		if ( ! is_wp_error( $tags ) ) {
			foreach ( $tags as $tag ) {
				$results[ $tag->slug ] = $tag->name;
			}
		}

		return $results;
	}

	protected function get_product_categories() {
		$categories = get_terms( array(
				'taxonomy'   => 'product_cat',
				'hide_empty' => false,
			)
		);
		$results    = array();
		if ( ! is_wp_error( $categories ) ) {
			foreach ( $categories as $category ) {
				$results[ $category->slug ] = $category->name;
			}
		}

		return $results;
	}

	protected function get_carousel_settings( $settings ) {
		return array(
			'dimensions'         => $settings['dimensions'],
			'navigation'         => $settings['navigation'],
			'autoplayHoverPause' => $settings['pause_on_hover'] === 'yes' ? true : false,
			'autoplay'           => $settings['autoplay'] === 'yes' ? true : false,
			'autoplayTimeout'    => $settings['autoplay_speed'],
			'autoplay_speed'     => $settings['autoplay_speed'],
			'items'              => $settings['column'],
			'items_laptop'       => isset($settings['column_laptop']) ? $settings['column_laptop'] : $settings['column'],
			'items_tablet'       => isset($settings['column_tablet']) ? $settings['column_tablet'] : $settings['column'],
			'items_mobile'       => isset($settings['column_mobile']) ? $settings['column_mobile'] : 1,
			'loop'               => $settings['infinite'] === 'yes' ? true : false,
			'gap'                => $settings['gap'], 
            'gap_laptop'         => isset($settings['gap_laptop']) ? $settings['gap_laptop'] : 20,
            'gap_tablet'         => isset($settings['gap_tablet']) ? $settings['gap_tablet'] : 15,
            'gap_mobile'         => isset($settings['gap_mobile']) ? $settings['gap_mobile'] : 10,
		);
	}

	protected function add_control_carousel( $condition = array() ) {
		$this->start_controls_section(
			'section_carousel_options',
			[
				'label'     => __( 'Carousel Options', 'templatemela-core' ),
				'type'      => Controls_Manager::SECTION,
				'condition' => $condition,
			]
		);

		$this->add_control(
			'enable_carousel',
			[
				'label' => __( 'Enable', 'templatemela-core' ),
				'type'  => Controls_Manager::SWITCHER,
			]
		);

		$this->add_control(
            'dimensions',
            [
                'label'          => esc_html__('Dimensions', 'templatemela-core'),
                'type'           => \Elementor\Controls_Manager::SELECT,
                'default'        => 1,
                'options'        => [1 => 1, 2 => 2, 3 => 3],
				'condition' => [
                    'enable_carousel' => 'yes'
                ],
            ]
        );

		$this->add_responsive_control(
            'gap',
            [
                'label'   => esc_html__('Slide Gap', 'templatemela-core'),
                'type'    => Controls_Manager::NUMBER,
                'default'        => 20,
                'laptop_default' => 20,
                'tablet_default' => 15,
                'mobile_default' => 10,
				'condition' => [
                    'enable_carousel' => 'yes'
                ],
            ]
        );

		$this->add_control(
			'navigation',
			[
				'label'     => __( 'Navigation', 'templatemela-core' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => 'arrows',
				'options'   => [
					'both'   => __( 'Arrows and Dots', 'templatemela-core' ),
					'arrows' => __( 'Arrows', 'templatemela-core' ),
					'dots'   => __( 'Dots', 'templatemela-core' ),
					'none'   => __( 'None', 'templatemela-core' ),
				],
				'condition' => [
					'enable_carousel' => 'yes'
				],
			]
		);

		// add dots bg color
        $this->add_control(
            'dots_bg_color',
            [
                'label'     => esc_html__('Dots Color', 'templatemela-core'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .splide .splide__pagination__page' => 'background: {{VALUE}};',
                ],
                'conditions' => [
                    'relation' => 'and',
                    'terms'    => [
                        [
                            'name'     => 'enable_carousel',
                            'operator' => '==',
                            'value'    => 'yes',
                        ],
                        [
                            'name'     => 'navigation',
                            'operator' => '!==',
                            'value'    => 'none',
                        ],
                        [
                            'name'     => 'navigation',
                            'operator' => '!==',
                            'value'    => 'arrows',
                        ],
                ],
                ],
            ]
        );

        $this->add_control(
            'dots_active_bg_color',
            [
                'label'     => esc_html__('Active Dots Color', 'templatemela-core'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .splide .splide__pagination__page.is-active' => 'background: {{VALUE}};',
                ],
                'conditions' => [
                    'relation' => 'and',
                    'terms'    => [
                        [
                            'name'     => 'enable_carousel',
                            'operator' => '==',
                            'value'    => 'yes',
                        ],
                        [
                            'name'     => 'navigation',
                            'operator' => '!==',
                            'value'    => 'none',
                        ],
                        [
                            'name'     => 'navigation',
                            'operator' => '!==',
                            'value'    => 'arrows',
                        ],
                ],
                ],
            ]
        );

		$this->add_control(
			'pause_on_hover',
			[
				'label'     => __( 'Pause on Hover', 'templatemela-core' ),
				'type'      => Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'condition' => [
					'enable_carousel' => 'yes'
				],
			]
		);

		$this->add_control(
			'autoplay',
			[
				'label'     => __( 'Autoplay', 'templatemela-core' ),
				'type'      => Controls_Manager::SWITCHER,
				'default'   => 'no',
				'condition' => [
					'enable_carousel' => 'yes'
				],
			]
		);

		$this->add_control(
			'autoplay_speed',
			[
				'label'     => __( 'Autoplay Speed', 'templatemela-core' ),
				'type'      => Controls_Manager::NUMBER,
				'default'   => 3000,
				'condition' => [
					'autoplay'        => 'yes',
					'enable_carousel' => 'yes'
				],
			]
		);

		$this->add_control(
			'infinite',
			[
				'label'     => __( 'Infinite Loop', 'templatemela-core' ),
				'type'      => Controls_Manager::SWITCHER,
				'default'   => 'no',
				'condition' => [
					'enable_carousel' => 'yes'
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'carousel_arrows',
			[
				'label'      => __( 'Carousel Arrows', 'templatemela-core' ),
				'conditions' => [
					'relation' => 'and',
					'terms'    => [
						[
							'name'     => 'enable_carousel',
							'operator' => '==',
							'value'    => 'yes',
						],
						[
							'name'     => 'navigation',
							'operator' => '!==',
							'value'    => 'none',
						],
						[
							'name'     => 'navigation',
							'operator' => '!==',
							'value'    => 'dots',
						],
					],
				],
			]
		);

		$this->add_control(
            'display_on_hover',
            [
                'label'     => esc_html__('Disaply on Hover', 'templatemela-core'),
                'type'      => Controls_Manager::SWITCHER,
                'default' => 'yes',
                'selectors'  => [
                    '{{WRAPPER}} .splide .splide__arrow' => 'opacity: 0;',
                    '{{WRAPPER}} .splide:hover .splide__arrow' => 'opacity: 1;',
                ],
            ]
        );

		$this->add_control(
			'next_heading',
			[
                'label' => esc_html__('Next button', 'templatemela-core'),
				'type'  => Controls_Manager::HEADING,
			]
		);

		$this->add_control(
			'next_vertical',
			[
                'label'       => esc_html__('Next Vertical', 'templatemela-core'),
				'type'        => Controls_Manager::CHOOSE,
				'label_block' => false,
				'options'     => [
					'top'    => [
                        'title' => esc_html__('Top', 'templatemela-core'),
						'icon'  => 'eicon-v-align-top',
					],
					'bottom' => [
                        'title' => esc_html__('Bottom', 'templatemela-core'),
						'icon'  => 'eicon-v-align-bottom',
					],
				],
				'default'     => 'top'
			]
		);

		$this->add_responsive_control(
			'next_vertical_value',
			[
				'type'       => Controls_Manager::SLIDER,
				'show_label' => false,
                'size_units' => ['px', '%'],
				'range'      => [
					'px' => [
                        'min'  => -1000,
						'max'  => 1000,
						'step' => 1,
					],
					'%'  => [
                        'min' => -100,
						'max' => 100,
					],
				],
				'default'    => [
					'unit' => '%',
					'size' => 35,
				],
				'selectors'  => [
                    '{{WRAPPER}} .splide .splide__arrow--next' => 'top: unset; bottom: unset; {{next_vertical.value}}: {{SIZE}}{{UNIT}};',
				]
			]
		);
		$this->add_control(
			'next_horizontal',
			[
                'label'       => esc_html__('Next Horizontal', 'templatemela-core'),
				'type'        => Controls_Manager::CHOOSE,
				'label_block' => false,
				'options'     => [
					'left'  => [
                        'title' => esc_html__('Left', 'templatemela-core'),
						'icon'  => 'eicon-h-align-left',
					],
					'right' => [
                        'title' => esc_html__('Right', 'templatemela-core'),
						'icon'  => 'eicon-h-align-right',
					],
				],
				'default'     => 'right'
			]
		);
		$this->add_responsive_control(
			'next_horizontal_value',
			[
				'type'       => Controls_Manager::SLIDER,
				'show_label' => false,
                'size_units' => ['px', 'em', '%'],
				'range'      => [
					'px' => [
                        'min'  => -1000,
						'max'  => 1000,
						'step' => 1,
					],
					'%'  => [
                        'min' => -100,
						'max' => 100,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors'  => [
                    '{{WRAPPER}} .splide .splide__arrow--next' => 'left: unset; right: unset;{{next_horizontal.value}}: {{SIZE}}{{UNIT}};',
				]
			]
		);


		$this->add_control(
			'prev_heading',
			[
                'label'     => esc_html__('Prev button', 'templatemela-core'),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_control(
			'prev_vertical',
			[
                'label'       => esc_html__('Prev Vertical', 'templatemela-core'),
				'type'        => Controls_Manager::CHOOSE,
				'label_block' => false,
				'options'     => [
					'top'    => [
                        'title' => esc_html__('Top', 'templatemela-core'),
						'icon'  => 'eicon-v-align-top',
					],
					'bottom' => [
                        'title' => esc_html__('Bottom', 'templatemela-core'),
						'icon'  => 'eicon-v-align-bottom',
					],
				],
				'default'     => 'top'
			]
		);

		$this->add_responsive_control(
			'prev_vertical_value',
			[
				'type'       => Controls_Manager::SLIDER,
				'show_label' => false,
                'size_units' => ['px', '%'],
				'range'      => [
					'px' => [
                        'min'  => -1000,
						'max'  => 1000,
						'step' => 1,
					],
					'%'  => [
                        'min' => -100,
						'max' => 100,
					],
				],
				'default'    => [
					'unit' => '%',
					'size' => 35,
				],
				'selectors'  => [
                    '{{WRAPPER}} .splide .splide__arrow--prev' => 'top: unset; bottom: unset; {{prev_vertical.value}}: {{SIZE}}{{UNIT}};',
				]
			]
		);
		$this->add_control(
			'prev_horizontal',
			[
                'label'       => esc_html__('Prev Horizontal', 'templatemela-core'),
				'type'        => Controls_Manager::CHOOSE,
				'label_block' => false,
				'options'     => [
					'left'  => [
                        'title' => esc_html__('Left', 'templatemela-core'),
						'icon'  => 'eicon-h-align-left',
					],
					'right' => [
                        'title' => esc_html__('Right', 'templatemela-core'),
						'icon'  => 'eicon-h-align-right',
					],
				],
				'default'     => 'left'
			]
		);
		$this->add_responsive_control(
			'prev_horizontal_value',
			[
				'type'       => Controls_Manager::SLIDER,
				'show_label' => false,
                'size_units' => ['px', 'em', '%'],
				'range'      => [
					'px' => [
                        'min'  => -1000,
						'max'  => 1000,
						'step' => 1,
					],
					'%'  => [
                        'min' => -100,
						'max' => 100,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' =>  0,
				],
				'selectors'  => [
                    '{{WRAPPER}} .splide .splide__arrow--prev' => 'left: unset; right: unset; {{prev_horizontal.value}}: {{SIZE}}{{UNIT}};',
				]
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_product_icon',
			[
				'label' => esc_html__( 'Carousel Arrows', 'templatemela-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
                'conditions' => [
                    'relation' => 'and',
                    'terms'    => [
                        [
                            'name'     => 'enable_carousel',
                            'operator' => '==',
                            'value'    => 'yes',
                        ],
                        [
                            'name'     => 'navigation',
                            'operator' => '!==',
                            'value'    => 'none',
                        ],
                        [
                            'name'     => 'navigation',
                            'operator' => '!==',
                            'value'    => 'dots',
                        ],
                    ],
                ],
			]
		);

        $this->start_controls_tabs( 'icon_style', [
            'condition' => $condition,
		] );

        $this->start_controls_tab(
			'icon_normal',
			[
				'label' => esc_html__( 'Normal', 'templatemela-core' ),
			]
		);

        $this->add_control(
			'icon_color',
			[
				'label' => esc_html__( 'Color', 'templatemela-core' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .splide .splide__arrow' => 'color: {{VALUE}};',
				],
			]
		);

        $this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name' => 'icon_bg_color',
				'types' => [ 'classic', 'gradient' ],
				'exclude' => [ 'image' ],
				'selector' => '{{WRAPPER}} .splide .splide__arrow',
				'fields_options' => [
					'background' => [
						'default' => 'classic',
					],
					'color' => [
						'global' => [
							'default' => '',
						],
					],
				],
			]
		);

		$this->end_controls_tab();

        $this->start_controls_tab(
			'icon_hover',
			[
				'label' => esc_html__( 'Hover', 'templatemela-core' ),
			]
		);

        $this->add_control(
			'icon_hover_color',
			[
				'label' => esc_html__( 'Text Color', 'templatemela-core' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .splide .splide__arrow:hover, {{WRAPPER}} .splide .splide__arrow:focus' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name' => 'icon_bg_hover_color',
				'types' => [ 'classic', 'gradient' ],
				'exclude' => [ 'image' ],
				'selector' => '{{WRAPPER}} .splide .splide__arrow:hover, {{WRAPPER}} .splide .splide__arrow:focus',
				'fields_options' => [
					'background' => [
						'default' => 'classic',
					],
				],
			]
		);

        $this->add_control(
			'icon_hover_border_color',
			[
				'label' => esc_html__( 'Border Color', 'templatemela-core' ),
				'type' => Controls_Manager::COLOR,
				'condition' => [
					'border_border!' => '',
				],
				'selectors' => [
					'{{WRAPPER}} .splide .splide__arrow:hover, {{WRAPPER}} .splide .splide__arrow:focus' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

        $this->end_controls_tabs();

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name' => 'border',
				'selector' => '{{WRAPPER}} .splide .splide__arrow',
				'separator' => 'before',
			]
		);

		$this->add_responsive_control(
			'icon_radius',
			[
				'label' => esc_html__( 'Border Radius', 'templatemela-core' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em', 'rem', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .splide .splide__arrow' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name' => 'icon_box_shadow',
				'selector' => '{{WRAPPER}} .splide .splide__arrow',
			]
		);

		$this->end_controls_section();
		

		// LIST STYLE
		$this->start_controls_section(
			'section_style_content',
			[
				'label' => esc_html__( 'List', 'templatemela-core' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);
		
		$this->add_control(
            'list_product_bg',
            [
                'label'     => __('Background Color', 'templatemela-core'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .products-list ul.products > li' => 'background: {{VALUE}};',
                ],
            ]
        );
		
		$this->add_responsive_control(
			'list_product_padding',
			[
				'label' => esc_html__( 'Padding', 'templatemela-core' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em', 'rem', 'vw', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .products-list ul.products > li' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		
		$this->add_responsive_control(
			'text_align',
			[
				'label' => esc_html__( 'Alignment', 'templatemela-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => esc_html__( 'Left', 'templatemela-core' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'templatemela-core' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => esc_html__( 'Right', 'templatemela-core' ),
						'icon' => 'eicon-text-align-right',
					],
				],
				'selectors' => [
					'{{WRAPPER}} .products-list ul.products > li' => 'text-align: {{VALUE}};',
				],
			]
		);
		
		$this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name'        => 'list_product_border',
                'placeholder' => '1px',
                'default'     => '1px',
                'selector'    => '{{WRAPPER}} .products-list ul.products > li',
                'separator'   => 'before',
            ]
        );

        $this->add_control(
            'list_product_radius',
            [
                'label'      => __('Border Radius', 'templatemela-core'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .products-list ul.products > li' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'list_product_box_shadow',
                'selector' => '{{WRAPPER}} .products-list ul.products > li',
            ]
        );

		$this->add_control(
			'list_title_heading',
			[
				'label' => esc_html__( 'Title', 'templatemela-core' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'tilte_typography',
                'selector' => '{{WRAPPER}} .products-list .product-title',
            ]
        );
		
		$this->add_control(
            'list_title_color',
            [
                'label' => __('Color', 'templatemela-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .products-list .product-title' => 'color: {{VALUE}};',
                ],
            ]
        );
		
		$this->add_control(
            'list_title_hover_color',
            [
                'label' => __('Hover Color', 'templatemela-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .products-list .product-title:hover' => 'color: {{VALUE}};',
                ],
            ]
        );
		
		 $this->add_responsive_control(
            'title_margin',
            [
                'label'      => __('Margin', 'templatemela-core'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors'  => [
                    '{{WRAPPER}} .products-list .product-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
		
		$this->add_control(
			'list_price_heading',
			[
				'label' => esc_html__( 'Price', 'templatemela-core' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'price_typography',
                'selector' => '{{WRAPPER}} .products-list .woocommerce-Price-amount',
            ]
        );
		
		$this->add_control(
            'price_color',
            [
                'label' => __('Color', 'templatemela-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .products-list .woocommerce-Price-amount' => 'color: {{VALUE}};',
                ],
            ]
        );

		$this->end_controls_section();
	}
}

$widgets_manager->register( new TemplateMelaCore_Elementor_Products_Tabs() );
