<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

if (!class_exists( 'WooCommerce' )) {
    return;
}

use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Background;

/**
 * Elementor tabs widget.
 *
 * Elementor widget that displays vertical or horizontal tabs with different
 * pieces of content.
 *
 * @since 1.0.0
 */
class TemplateMelaCore_Elementor_Product_Categories extends \Elementor\Widget_Base {


    public function get_categories() {
        return array('tmcore-addons');
    }

    /**
     * Get widget name.
     *
     * Retrieve tabs widget name.
     *
     * @return string Widget name.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_name() {
        return 'tmcore-product-categories-carousel';
    }

    /**
     * Get widget title.
     *
     * Retrieve tabs widget title.
     *
     * @return string Widget title.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_title() {
        return esc_html__('TM Product Categories', 'templatemela-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve tabs widget icon.
     *
     * @return string Widget icon.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_icon() {
        return 'eicon-carousel-loop';
    }


    public function get_script_depends() {
        return [
            'tmcore-elementor-product-categories',
            'base-splide',
        ];
    }

    /**
     * Register tabs widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since  1.0.0
     * @access protected
     */
    protected function register_controls() {

        //Section Settings
        $this->start_controls_section(
            'section_setting',
            [
                'label' => esc_html__('Settings', 'templatemela-core'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'categories',
            [
                'label'       => esc_html__('Categories', 'templatemela-core'),
                'type'        => Controls_Manager::SELECT2,
                'options'     => $this->get_product_categories(),
                'label_block' => true,
                'multiple'    => true,
            ]
        );

        $this->add_responsive_control(
            'column',
            [
                'label'          => esc_html__('Columns', 'templatemela-core'),
                'type'           => \Elementor\Controls_Manager::SELECT,
                'default'        => 6,
                'laptop_default' => 5,
                'tablet_default' => 3,
                'mobile_default' => 2,
                'options'        => [1 => 1, 2 => 2, 3 => 3, 4 => 4, 5 => 5, 6 => 6, 7 => 7, 8 => 8, 9 => 9, 10 => 10, 11 => 11, 12 => 12, 13 => 13, 14 => 14, 15 => 15],
            ]
        );

        $this->add_control(
            'column_minimum',
            [
                'label'     => esc_html__('Columns Below 540', 'templatemela-core'),
                'description' => esc_html__( 'Columns to Show on devices below 540. Fix for slider with small categories', 'templatemela-core' ),
                'type'      => Controls_Manager::SELECT,
                'default'   => 2,
                'options'        => [1 => 1, 2 => 2, 3 => 3, 4 => 4, 5 => 5],
            ]
        );

        $this->add_control(
            'advanced',
            [
                'label'     => esc_html__('Advanced', 'templatemela-core'),
                'type'      => Controls_Manager::HEADING,
               
            ]
        );

        $this->add_control(
            'orderby',
            [
                'label'     => esc_html__('Order By', 'templatemela-core'),
                'type'      => Controls_Manager::SELECT,
                'default'   => 'include',
                'options'   => [
                    'name'       => esc_html__('Name', 'templatemela-core'),
                    'id'         => esc_html__('Category ID', 'templatemela-core'),
                    'title'      => esc_html__('Category Title', 'templatemela-core'),
                    'include'    => esc_html__('Same as Include', 'templatemela-core'),
                    'order'      => esc_html__('Default Order', 'templatemela-core'),                    
                ]
            ]
        );

        $this->add_control(
            'order',
            [
                'label'     => esc_html__('Order', 'templatemela-core'),
                'type'      => Controls_Manager::SELECT,
                'default'   => 'desc',
                'options'   => [
                    'asc'  => esc_html__('ASC', 'templatemela-core'),
                    'desc' => esc_html__('DESC', 'templatemela-core'),
                ]
            ]
        );


        /* $this->add_control(
            'categories_style',
            [
                'label'   => esc_html__('Category Layout', 'templatemela-core'),
                'type'    => Controls_Manager::SELECT,
                'default' => 'grid',
                'options' => [
                    'grid' => esc_html__('Grid', 'templatemela-core'),
                    'list' => esc_html__('List', 'templatemela-core'),
                ],
            ]
        ); */

        $this->add_control(
            'categories_style',
            [
                'label'     => esc_html__('Style', 'templatemela-core'),
                'type'      => Controls_Manager::SELECT,
                'default'   => '1',
                'options'   => [
                    '1' => esc_html__('Style 1', 'templatemela-core'),
                    '2' => esc_html__('Style 2', 'templatemela-core'),
                    '3' => esc_html__('Style 3', 'templatemela-core'),
                    '4' => esc_html__('Style 4', 'templatemela-core'),
                    '5' => esc_html__('Style 5', 'templatemela-core'),
                    '6' => esc_html__('Style 6', 'templatemela-core'),
                ],
                'prefix_class' => 'product-cat-style-',
            ]
        );

        /* $this->add_control(
            'list_style',
            [
                'label'     => esc_html__('List Style', 'templatemela-core'),
                'type'      => Controls_Manager::SELECT,
                'default'   => '1',
                'options'   => [
                    '1' => esc_html__('Style 1', 'templatemela-core'),
                    '2' => esc_html__('Style 2', 'templatemela-core'),
                    '3' => esc_html__('Style 3', 'templatemela-core'),
                    '4' => esc_html__('Style 4', 'templatemela-core'),
                    '5' => esc_html__('Style 5', 'templatemela-core'),
                ],
                'prefix_class' => 'product-cat-style-',
                'condition' => [
                    'categories_style' => 'list'
                ]
            ]
        ); */

        /* $this->add_responsive_control(
            'list_image_size',
            [
                'label' => esc_html__( 'Image Width', 'templatemela-core' ) . ' (%)',
                'type'       => Controls_Manager::SLIDER,
                'default' => [
					'size' => 30,
					'unit' => '%',
				],
				'laptop_default' => [
					'unit' => '%',
				],
				'tablet_default' => [
					'unit' => '%',
				],
				'mobile_default' => [
					'unit' => '%',
				],
				'size_units' => [ 'px', '%', 'em', 'rem', 'vw', 'custom' ],
				'range' => [
					'%' => [
						'min' => 5,
						'max' => 100,
					],
				],
                'selectors'  => [
                    '{{WRAPPER}} .categories-wrap li.category image-wrap'      =>  'width: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'categories_style' => 'list'
                ]
            ]
        ); */


        $this->add_control(
            'show_image',
            [
                'label' => __('Show Image', 'templatemela-core'),
                'type'  => Controls_Manager::SWITCHER,
                'default'   => 'yes',
            ]
        );

        $this->add_control(
            'show_product_count',
            [
                'label'      => esc_html__('Show Product Count', 'templatemela-core'),
                'type'       => Controls_Manager::SWITCHER,
                'default'   => 'yes',
            ]
        );

        $this->add_control(
            'show_description',
            [
                'label'      => esc_html__('Show Description', 'templatemela-core'),
                'type'       => Controls_Manager::SWITCHER,
                'default'   => 'no',
            ]
        );

        $this->add_control(
            'show_sub_categories',
            [
                'label'      => esc_html__('Show Sub Categories', 'templatemela-core'),
                'type'       => Controls_Manager::SWITCHER,
                'default'   => 'no',
            ]
        );

        $this->add_control(
            'sub_category_limit',
            [
                'label' => esc_html__('Sub Category Limit', 'templatemela-core'),
                'type' => Controls_Manager::NUMBER,
                'default' => 3,
                'condition'   => [
                    'show_sub_categories' => 'yes'
                ]
            ]
        );

        /* $this->add_control(
            'show_button',
            [
                'label'      => esc_html__('Show Button', 'templatemela-core'),
                'type'       => Controls_Manager::SWITCHER,
                'default'   => 'no',
            ]
        );
        
        $this->add_control(
            'button_text',
            [
                'label' => esc_html__('Button Text', 'templatemela-core'),
                'default' => 'Shop Now',
                'type' => Controls_Manager::TEXT,
                'condition'   => [
                    'show_button' => 'yes'
                ]
            ]
        ); */


        $this->end_controls_section();
        // End Section Settings


        // WRAPPER STYLE
        $this->start_controls_section(
            'section_style_category_wrapper',
            [
                'label' => __('Wrapper', 'templatemela-core'),
                'tab'   => Controls_Manager::TAB_STYLE,

            ]
        );

        $this->add_responsive_control(
			'category_align',
			[
				'label' => esc_html__( 'Alignment', 'templatemela-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => esc_html__( 'Left', 'templatemela-core' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'templatemela-core' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => esc_html__( 'Right', 'templatemela-core' ),
						'icon' => 'eicon-text-align-right',
					],
				],
				'selectors' => [
					'{{WRAPPER}} .product-cat' => 'text-align: {{VALUE}}',
				],
			]
		);

        $this->add_responsive_control(
            'padding_category_wrapper',
            [
                'label'      => __('Padding', 'templatemela-core'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .product-cat' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
			'category_gutter',
			[
				'label'      => __( 'Spacing', 'templatemela-core' ),
				'type'       => Controls_Manager::SLIDER,
				'default'    =>  [
					'size' => 20,
					'unit' => 'px',
				],
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'size_units' => [ 'px', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} ul.categories li.category'           => 'margin-left: calc({{SIZE}}{{UNIT}} / 2); margin-right: calc({{SIZE}}{{UNIT}} / 2); margin-bottom: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} ul.categories li.category-item'      => 'margin-left: calc({{SIZE}}{{UNIT}} / 2); margin-right: calc({{SIZE}}{{UNIT}} / 2); margin-bottom: calc({{SIZE}}{{UNIT}} - 1px);',
					'{{WRAPPER}} .categories-wrap ul.categories > li'  => 'margin-left: calc({{SIZE}}{{UNIT}} / 2); margin-right: calc({{SIZE}}{{UNIT}} / 2); margin-bottom: calc({{SIZE}}{{UNIT}} - 1px);',
					'{{WRAPPER}} ul.categories'                      => 'margin-left: calc({{SIZE}}{{UNIT}} / -2); margin-right: calc({{SIZE}}{{UNIT}} / -2);',
				],
                'condition'   => [
                    'enable_carousel!' => 'yes'
                ]
			]
		);

        /* $this->add_responsive_control(
            'margin_category_wrapper',
            [
                'label'      => __('Margin', 'templatemela-core'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors'  => [
                    '{{WRAPPER}} .product-cat' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'enable_carousel!' => 'yes'
                ],
            ]
        ); */

        $this->add_control(
            'color_category_wrapper',
            [
                'label'     => __('Background Color', 'templatemela-core'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .product-cat' => 'background: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name'        => 'category_wrapper_border',
                'placeholder' => '1px',
                'default'     => '1px',
                'selector'    => '{{WRAPPER}} .product-cat',
                'separator'   => 'before',
            ]
        );

        $this->add_control(
            'category_wrapper_radius',
            [
                'label'      => __('Border Radius', 'templatemela-core'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .product-cat' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'category_wrapper_box_shadow',
                'selector' => '{{WRAPPER}} .product-cat',
            ]
        );

        $this->end_controls_section();
        // End Wrapper Style

        // Image Style
        $this->start_controls_section(
            'category_image_style',
            [
                'label' => __('Image', 'templatemela-core'),
                'tab'   => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'show_image' => 'yes'
                ],
            ]
        );
		
        $this->add_responsive_control(
			'category_image_width',
			[
				'label' => esc_html__( 'Width', 'templatemela-core' ) . ' (%)',
				'type' => Controls_Manager::SLIDER,
                'condition' => [
                    'show_image' => 'yes',
                    /* 'categories_style' => [
                        '5',
                        '6',
                    ], */
                ],
				'default' => [
					'size' => 30,
					'unit' => '%',
				],
				'tablet_default' => [
					'unit' => '%',
				],
				'mobile_default' => [
					'unit' => '%',
				],
				'size_units' => [ '%' ],
				'range' => [
					'%' => [
						'min' => 5,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .cat-image' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);

        $this->add_responsive_control(
            'category_image_margin',
            [
                'label'      => __('Margin', 'templatemela-core'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors'  => [
                    '{{WRAPPER}} .product-cat .cat-image' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'show_image' => 'yes'
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name'        => 'category_image_border',
                'placeholder' => '1px',
                'default'     => '1px',
                'selector'    => '{{WRAPPER}} .product-cat .cat-image img, .woocommerce {{WRAPPER}} ul.products li.product .woocommerce-placeholder',
                'separator'   => 'before',
                'condition' => [
                    'show_image' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'category_image_radius',
            [
                'label'      => __('Image Radius', 'templatemela-core'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .product-cat .cat-image img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'show_image' => 'yes'
                ],
            ]
        );

        $this->end_controls_section();
        // End Image Style

        // Title Style
        $this->start_controls_section(
            'title_style',
            [
                'label' => __('Title', 'templatemela-core'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'tilte_typography',
                'selector' => '{{WRAPPER}} .cat-title',
            ]
        );

        $this->start_controls_tabs('tab_title');
        $this->start_controls_tab(
            'tab_title_normal',
            [
                'label' => __('Normal', 'templatemela-core'),
            ]
        );
        $this->add_control(
            'title_color',
            [
                'label' => __('Color', 'templatemela-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .cat-title a' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'title_background',
            [
                'label' => __('Background', 'templatemela-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .cat-title ' => 'background: {{VALUE}};',
                ],
            ]
        );
        $this->end_controls_tab();
        $this->start_controls_tab(
            'tab_title_hover',
            [
                'label' => __('Hover', 'templatemela-core'),
            ]
        );
        $this->add_control(
            'title_color_hover',
            [
                'label' => __('Hover Color', 'templatemela-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .cat-title:hover a ' => 'color: {{VALUE}};',
                ],
            ]
        );
        $this->add_control(
            'title_background_hover',
            [
                'label' => __('Background Hover', 'templatemela-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .cat-title:hover ' => 'background: {{VALUE}};',
                ],
            ]
        );
        $this->end_controls_tab();
        $this->end_controls_tabs();


        $this->add_responsive_control(
            'title_padding',
            [
                'label'      => __( 'Padding', 'templatemela-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'separator' => 'before',
                'selectors'  => [
                    '{{WRAPPER}} .cat-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        
        $this->end_controls_section();
        // End Title  style

        // Product Count - Total
        $this->start_controls_section(
            'total_style',
            [
                'label' => __('Product Count', 'templatemela-core'),
                'tab'   => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'show_product_count' => 'yes'
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'total_typography',
                'selector' => '{{WRAPPER}} .cat-total',
                'condition' => [
                    'show_product_count' => 'yes'
                ],
            ]
        );

        $this->start_controls_tabs('tab_total');
        $this->start_controls_tab(
            'tab_total_normal',
            [
                'label' => __('Normal', 'templatemela-core'),
                'condition' => [
                    'show_product_count' => 'yes'
                ],
            ]
        );
        $this->add_control(
            'total_color',
            [
                'label' => __('Color', 'templatemela-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .cat-total' => 'color: {{VALUE}};',
                ],
                'condition' => [
                    'show_product_count' => 'yes'
                ],
            ]
        );
        $this->add_control(
            'total_background',
            [
                'label' => __('Background', 'templatemela-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .cat-total ' => 'background: {{VALUE}};',
                ],
                'condition' => [
                    'show_product_count' => 'yes'
                ],
            ]
        );
        $this->end_controls_tab();


        $this->start_controls_tab(
            'tab_total_hover',
            [
                'label' => __('Hover', 'templatemela-core'),
                'condition' => [
                    'show_product_count' => 'yes'
                ],
            ]
        );
        $this->add_control(
            'total_color_hover',
            [
                'label' => __('Color Hover', 'templatemela-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .product-cat:hover .cat-total' => 'color: {{VALUE}};',
                ],
                'condition' => [
                    'show_product_count' => 'yes'
                ],
            ]
        );
        $this->add_control(
            'total_background_hover',
            [
                'label' => __('Background Hover', 'templatemela-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .product-cat:hover .cat-total ' => 'background: {{VALUE}};',
                ],
                'condition' => [
                    'show_product_count' => 'yes'
                ],
            ]
        );
        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_responsive_control(
            'total_padding',
            [
                'label'      => __( 'Padding', 'templatemela-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'separator' => 'before',
                'selectors'  => [
                    '{{WRAPPER}} .cat-total' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'show_product_count' => 'yes'
                ],
            ]
        );

        $this->end_controls_section();
        // End product total count

        // sub categories style
        $this->start_controls_section(
            'sub_category_style',
            [
                'label' => __('Sub Categories', 'templatemela-core'),
                'tab'   => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'show_sub_categories' => 'yes'
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'sub_category_typography',
                'selector' => '{{WRAPPER}} .sub-categories li a',
                'condition' => [
                    'show_sub_categories' => 'yes'
                ],
            ]
        );

        $this->start_controls_tabs('sub_category');

        $this->start_controls_tab(
            'sub_category_normal',
            [
                'label' => __('Normal', 'templatemela-core'),
                'condition' => [
                    'show_sub_categories' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'sub_category_color',
            [
                'label' => __('Color', 'templatemela-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .sub-categories li a' => 'color: {{VALUE}};',
                ],
                'condition' => [
                    'show_sub_categories' => 'yes'
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'sub_category_hover',
            [
                'label' => __('Hover', 'templatemela-core'),
                'condition' => [
                    'show_sub_categories' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'sub_category_color_hover',
            [
                'label' => __('Color Hover', 'templatemela-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .sub-categories li a:hover' => 'color: {{VALUE}};',
                ],
                'condition' => [
                    'show_sub_categories' => 'yes'
                ],
            ]
        );
        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();
        // End sub categories style

        // Carousel Option
        $this->add_control_carousel();
    }

    protected function get_product_categories() {
        $categories = get_terms(array(
                'taxonomy'   => 'product_cat',
                'hide_empty' => false,
            )
        );
        $results    = array();
        if (!is_wp_error($categories)) {
            foreach ($categories as $category) {
                $results[$category->term_id] = $category->name;
            }
        }

        return $results;
    }

    /**
     * Render tabs widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since  1.0.0
     * @access protected
     */
    protected function render() {
        $settings = $this->get_settings_for_display();

        if (!empty($settings['categories']) && is_array($settings['categories'])) {

            $atts = [
                'category'         => $settings['categories'],
                'columns'          => isset($settings['column']) ? $settings['column'] : 6 ,
                'column_laptop'    => isset($settings['column_laptop']) ? $settings['column_laptop'] : 4 ,
                'column_tablet'    => isset($settings['column_tablet']) ? $settings['column_tablet'] : 3 ,
                'column_mobile'    => isset($settings['column_mobile']) ? $settings['column_mobile'] : 2 ,
                'orderby'          => $settings['orderby'],
                'order'            => $settings['order'],
                'categories_style' => $settings['categories_style'],
            ];
    
            $class = '';
            
            // Carousel
            if ($settings['enable_carousel'] === 'yes') {
                $atts['carousel_settings'] = wp_json_encode(wp_slash($this->get_carousel_settings()));
                $class .= ' carousel';
            } else {
                if (!empty($atts['column_laptop'])) {
                    $class .= ' columns-laptop-' . $atts['column_laptop'];
                }
                if (!empty($atts['column_tablet'])) {
                    $class .= ' columns-tablet-' . $atts['column_tablet'];
                }
                if (!empty($atts['column_mobile'])) {
                    $class .= ' columns-mobile-' . $atts['column_mobile'];
                }
            }
            $atts['class'] = $class;
            
            $term_names = get_terms( array(
                'taxonomy'   => 'product_cat',
                'include'    => $atts['category'],
                'hide_empty' => false, 
                'orderby'    => $settings['orderby'],
            ) );
    
            echo '<div class="woocommerce categories-wrap columns-' . $atts['columns'] . $atts['class'] . '">';
                
                $is_splide = '';
                if ($settings['enable_carousel'] === 'yes') {
                    echo '<div class="woocommerce-carousel splide" data-settings=\'' . $atts['carousel_settings'] . '\'>';
                    echo '<div class="splide__track">';
                    $is_splide = 'splide__list';
                }

                echo '<ul class="products categories content-wrap grid-cols ' . $is_splide . '">';
                foreach ( $term_names as $the_term ) {
                    
                    $count = 0;
                    
                    $term = get_term_by('ID', $the_term->term_id, 'product_cat');
                    $term_data = get_option('taxonomy_'.$term->term_id);
                    $thumbnail_id = get_term_meta( $term->term_id, 'thumbnail_id', true );
                    $term_children = get_term_children( $term->term_id, 'product_cat' );
                    
                    $category_name = empty($term->name) ? esc_html($term->name) : $term->name;
                    $catTitle = '<div class="cat-title"><a href="'. esc_url( get_term_link($term) ) . '" title="'. esc_attr( $term->name ) . '">' . $category_name . '</a></div>';

                    // Categories image
                    $catImage = '';
                    if ( 'yes' == $settings['show_image'] ) {

                        // Category image
                        $image_alt = get_post_meta( $thumbnail_id, '_wp_attachment_image_alt', true);
                        if ( empty( $image_alt )) {
                            $image_alt = $term->name;
                        }
                        $image = wp_get_attachment_image( $thumbnail_id, 'full', false, array( 'alt' => $image_alt ) );

                        $catImage .= '<a class="image-wrap link_category_product" href="'.esc_url( get_term_link( $term ) ).'" title="'.esc_attr($term->name).'" >';
                        $catImage .= ( !empty( $image ) ) ? $image : wc_placeholder_img();
                        $catImage .= '</a>';
                    }

                    // Products count
                    $proCount = '';
                    if ( 'yes' == $settings['show_product_count'] ) {
                        $proCount = '<div class="cat-total">'.esc_html( $term->count ).' ' . esc_html__('Products','templatemela-core') . '</div>';
                    }

                    // Categories description
                    $catDec = '';
                    if ( 'yes' == $settings['show_description'] ) {
                        $catDec = '<p class="category-description">'.esc_html( $term->description ).'</p>';
                    }

                    // Shop now button for categories
                    /* $button = '';
                    $showNowText =  !empty( $settings['button_text'] ) ? $settings['button_text'] : esc_html__('Shop Now','templatemela-core');
                    if ( 'yes' == $settings['show_button'] ) {
                        $button .= '<button class="button shop-now" href="'.esc_url( get_term_link( $term ) ).'">';
                        $button .= '<span class="shop-now-text">'. $showNowText .'</span>';
                        $button .= '</button>';
                    } */
                     
                    // Sub Categories
                    $showsubHTML = '';
                    if ( 'yes' == $settings['show_sub_categories'] ){

                        $cat_args = array(
                            'taxonomy'   => "product_cat",
                            'orderby'    => 'name',
                            'order'      => 'ASC',
                            'hide_empty' => false,
                            'parent'     => $term->term_id, // set your parent term id
                        );
                        $categories = get_terms( $cat_args );
                        $showsubHTML = '<ul class="sub-categories">';
                        foreach($categories as $category) {    
                            $showsubHTML .= '<li><a href="' . get_category_link( $category->term_id ) . '" title="' . sprintf( __( "View all products in %s" ), $category->name ) . '" ' . '>' . $category->name . '</a></li> ';
                            $count++; 
                            if ($count == $settings['sub_category_limit']){
                                break;
                            }               
                        }
                        $showsubHTML .= '</ul>';
                    }
                    
                    echo '<li class="shimmercat product category product-cat splide__slide">';
                    ?>
                        
                        <?php if ($settings['categories_style'] == '1'): ?>

                            <div class="cat-image">
                                <?php if ($settings['show_image']) : ?>
                                    <?php echo $catImage; ?>
                                <?php endif; ?>
                                <?php if ($settings['show_product_count']): ?>
                                    <?php echo $proCount; // products count ?>
                                <?php endif; ?>
                            </div>
                            
                            <?php echo $catTitle; ?>
            
                            <?php echo $catDec; ?>

                            <?php if ($settings['show_sub_categories']): ?>
                                <?php echo $showsubHTML; // Subcategories ?>
                            <?php endif; ?>
        
                            <?php elseif ($settings['categories_style'] == '2' || $settings['categories_style'] == '4'): ?>
                            
                            <?php if ($settings['show_image']) : ?>
                            <div class="cat-content">
                            <div class="cat-image">
                                <?php echo $catImage; ?>
                            </div>
                            <?php endif; ?>
                            <div class="product-cat-caption">
                                <?php echo $catTitle; ?>
                                <?php if ($settings['show_product_count']): ?>
                                    <?php echo $proCount; // products count ?>
                                <?php endif; ?>
                            </div>
                            </div>
        
                            <?php if ($settings['show_sub_categories']): ?>
                                <?php echo $showsubHTML; // Subcategories ?>
                            <?php endif; ?>

                        <?php elseif ($settings['categories_style'] == '5'): ?>
                            <div class="cat-contents">
                            <div class="cat-image">
                                <?php if ($settings['show_image']) : ?>
                                    <?php echo $catImage; ?>
                                <?php endif; ?>
                                </div>
                                <div class="product-cat-caption">
                                    <?php echo $catTitle; ?>
                                <?php if ($settings['show_product_count']): ?>
                                    <?php echo $proCount; // products count ?>
                                <?php endif; ?>
                                <?php echo $catDec; ?>
                            </div>
                            </div>
        
                            <?php if ($settings['show_sub_categories']): ?>
                                <?php echo $showsubHTML; // Subcategories ?>
                            <?php endif; ?>
        
                        <?php elseif ($settings['categories_style'] == '6'): ?>

                            <div class="cat-contents">
                                <div class="cat-image">
                                    <?php if ($settings['show_image']) : ?>
                                        <?php echo $catImage; ?>
                                    <?php endif; ?>
                                </div>
                                <div class="product-cat-caption">
                                <?php echo $catTitle; ?>
                                    <?php if ($settings['show_product_count']): ?>
                                        <?php echo $proCount; // products count ?>
                                    <?php endif; ?>
                                    <?php echo $catDec; ?>
                                </div>
                            </div>
                            
                                <?php if ($settings['show_sub_categories']): ?>
                                    <?php echo $showsubHTML; // Subcategories ?>
                                <?php endif; ?>
        
                        <?php else: ?>

                            <div class="cat-image">
                                <?php if ($settings['show_image']) : ?>
                                    <?php echo $catImage; ?>
                                <?php endif; ?>
                            </div>
                            <div class="cat-contents">
                                <?php echo $catTitle; ?>
                                <?php if ($settings['show_product_count']): ?>
                                    <?php echo $proCount; // products count ?>
                                <?php endif; ?>
                                <?php echo $catDec; ?>
                                <?php if ($settings['show_sub_categories']): ?>
                                    <?php echo $showsubHTML; // Subcategories ?>
                                <?php endif; ?>
                            </div>
                            
                        <?php endif; ?>
                    
                    <?php
                    
                    echo '</li>';

                } // end foreach
                echo '</ul>';

                if ($settings['enable_carousel'] === 'yes') {
                        echo '</div>';
                    echo '</div>';
                }
            
            echo '</div>';

        }
    }

    protected function add_control_carousel($condition = array()) {
        $this->start_controls_section(
            'section_carousel_options',
            [
                'label'     => esc_html__('Carousel Options', 'templatemela-core'),
                'type'      => Controls_Manager::SECTION,
                'condition' => $condition,
            ]
        );

        $this->add_control(
            'enable_carousel',
            [
                'label' => esc_html__('Enable', 'templatemela-core'),
                'type'  => Controls_Manager::SWITCHER,
            ]
        );

        $this->add_control(
            'dimensions',
            [
                'label'          => esc_html__('Dimensions', 'templatemela-core'),
                'type'           => \Elementor\Controls_Manager::SELECT,
                'default'        => 1,
                'options'        => [1 => 1, 2 => 2, 3 => 3],
                'condition' => [
                    'enable_carousel' => 'yes'
                ],
            ]
        );

        $this->add_responsive_control(
            'gap',
            [
                'label'   => esc_html__('Slide Gap', 'templatemela-core'),
                'type'    => Controls_Manager::NUMBER,
                'default'        => 20,
                'laptop_default' => 20,
                'tablet_default' => 15,
                'mobile_default' => 10,
                'condition' => [
                    'enable_carousel' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'navigation',
            [
                'label'     => esc_html__('Navigation', 'templatemela-core'),
                'type'      => Controls_Manager::SELECT,
                'default'   => 'arrows',
                'options'   => [
                    'both'   => esc_html__('Arrows and Dots', 'templatemela-core'),
                    'arrows' => esc_html__('Arrows', 'templatemela-core'),
                    'dots'   => esc_html__('Dots', 'templatemela-core'),
                    'none'   => esc_html__('None', 'templatemela-core'),
                ],
                'condition' => [
                    'enable_carousel' => 'yes'
                ],
            ]
        );

        // add dots bg color
        $this->add_control(
            'dots_bg_color',
            [
                'label'     => esc_html__('Dots Color', 'templatemela-core'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .splide .splide__pagination__page' => 'background: {{VALUE}};',
                ],
                'conditions' => [
                    'relation' => 'and',
                    'terms'    => [
                        [
                            'name'     => 'enable_carousel',
                            'operator' => '==',
                            'value'    => 'yes',
                        ],
                        [
                            'name'     => 'navigation',
                            'operator' => '!==',
                            'value'    => 'none',
                        ],
                        [
                            'name'     => 'navigation',
                            'operator' => '!==',
                            'value'    => 'arrows',
                        ],
                ],
                ],
            ]
        );

        $this->add_control(
            'dots_active_bg_color',
            [
                'label'     => esc_html__('Active Dots Color', 'templatemela-core'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .splide .splide__pagination__page.is-active' => 'background: {{VALUE}};',
                ],
                'conditions' => [
                    'relation' => 'and',
                    'terms'    => [
                        [
                            'name'     => 'enable_carousel',
                            'operator' => '==',
                            'value'    => 'yes',
                        ],
                        [
                            'name'     => 'navigation',
                            'operator' => '!==',
                            'value'    => 'none',
                        ],
                        [
                            'name'     => 'navigation',
                            'operator' => '!==',
                            'value'    => 'arrows',
                        ],
                ],
                ],
            ]
        );

        $this->add_control(
            'pause_on_hover',
            [
                'label'     => esc_html__('Pause on Hover', 'templatemela-core'),
                'type'      => Controls_Manager::SWITCHER,
                'default'   => 'yes',
                'condition' => [
                    'enable_carousel' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'autoplay',
            [
                'label'     => esc_html__('Autoplay', 'templatemela-core'),
                'type'      => Controls_Manager::SWITCHER,
                'default'   => 'no',
                'condition' => [
                    'enable_carousel' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'autoplay_speed',
            [
                'label'     => esc_html__('Autoplay Speed', 'templatemela-core'),
                'type'      => Controls_Manager::NUMBER,
                'default'   => 3000,
                'condition' => [
                    'autoplay'        => 'yes',
                    'enable_carousel' => 'yes'
                ],
                'frontend_available' => true,
            ]
        );

        $this->add_control(
            'infinite',
            [
                'label'     => esc_html__('Infinite Loop', 'templatemela-core'),
                'type'      => Controls_Manager::SWITCHER,
                'default'   => 'no',
                'condition' => [
                    'enable_carousel' => 'yes'
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'carousel_arrows',
            [
                'label'      => esc_html__('Carousel Arrows', 'templatemela-core'),
                'conditions' => [
                    'relation' => 'and',
                    'terms'    => [
                        [
                            'name'     => 'enable_carousel',
                            'operator' => '==',
                            'value'    => 'yes',
                        ],
                        [
                            'name'     => 'navigation',
                            'operator' => '!==',
                            'value'    => 'none',
                        ],
                        [
                            'name'     => 'navigation',
                            'operator' => '!==',
                            'value'    => 'dots',
                        ],
                    ],
                ],
            ]
        );

        $this->add_control(
            'display_on_hover',
            [
                'label'     => esc_html__('Disaply on Hover', 'templatemela-core'),
                'type'      => Controls_Manager::SWITCHER,
                'default' => 'yes',
                'selectors'  => [
                    '{{WRAPPER}} .splide .splide__arrow' => 'opacity: 0;',
                    '{{WRAPPER}} .splide:hover .splide__arrow' => 'opacity: 1;',
                ],
            ]
        );

        $this->add_control(
            'next_heading',
            [
                'label' => esc_html__('Next button', 'templatemela-core'),
                'type'  => Controls_Manager::HEADING,
            ]
        );

        $this->add_control(
            'next_vertical',
            [
                'label'       => esc_html__('Next Vertical', 'templatemela-core'),
                'type'        => Controls_Manager::CHOOSE,
                'label_block' => false,
                'options'     => [
                    'top'    => [
                        'title' => esc_html__('Top', 'templatemela-core'),
                        'icon'  => 'eicon-v-align-top',
                    ],
                    'bottom' => [
                        'title' => esc_html__('Bottom', 'templatemela-core'),
                        'icon'  => 'eicon-v-align-bottom',
                    ],
                ],
                'default'     => 'top'
            ]
        );

        $this->add_responsive_control(
            'next_vertical_value',
            [
                'type'       => Controls_Manager::SLIDER,
                'show_label' => false,
                'size_units' => ['px', '%'],
                'range'      => [
                    'px' => [
                        'min'  => -1000,
                        'max'  => 1000,
                        'step' => 1,
                    ],
                    '%'  => [
                        'min' => -100,
                        'max' => 100,
                    ],
                ],
                'default'    => [
                    'unit' => '%',
                    'size' => 35,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .splide .splide__arrow--next' => 'top: unset; bottom: unset; {{next_vertical.value}}: {{SIZE}}{{UNIT}};',
                ]
            ]
        );
        $this->add_control(
            'next_horizontal',
            [
                'label'       => esc_html__('Next Horizontal', 'templatemela-core'),
                'type'        => Controls_Manager::CHOOSE,
                'label_block' => false,
                'options'     => [
                    'left'  => [
                        'title' => esc_html__('Left', 'templatemela-core'),
                        'icon'  => 'eicon-h-align-left',
                    ],
                    'right' => [
                        'title' => esc_html__('Right', 'templatemela-core'),
                        'icon'  => 'eicon-h-align-right',
                    ],
                ],
                'default'     => 'right'
            ]
        );
        $this->add_responsive_control(
            'next_horizontal_value',
            [
                'type'       => Controls_Manager::SLIDER,
                'show_label' => false,
                'size_units' => ['px', 'em', '%'],
                'range'      => [
                    'px' => [
                        'min'  => -1000,
                        'max'  => 1000,
                        'step' => 1,
                    ],
                    '%'  => [
                        'min' => -100,
                        'max' => 100,
                    ],
                ],
                'default'    => [
                    'unit' => 'px',
                    'size' => 0,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .splide .splide__arrow--next' => 'left: unset; right: unset;{{next_horizontal.value}}: {{SIZE}}{{UNIT}};',
                ]
            ]
        );


        $this->add_control(
            'prev_heading',
            [
                'label'     => esc_html__('Prev button', 'templatemela-core'),
                'type'      => Controls_Manager::HEADING,
                'separator' => 'before'
            ]
        );

        $this->add_control(
            'prev_vertical',
            [
                'label'       => esc_html__('Prev Vertical', 'templatemela-core'),
                'type'        => Controls_Manager::CHOOSE,
                'label_block' => false,
                'options'     => [
                    'top'    => [
                        'title' => esc_html__('Top', 'templatemela-core'),
                        'icon'  => 'eicon-v-align-top',
                    ],
                    'bottom' => [
                        'title' => esc_html__('Bottom', 'templatemela-core'),
                        'icon'  => 'eicon-v-align-bottom',
                    ],
                ],
                'default'     => 'top'
            ]
        );

        $this->add_responsive_control(
            'prev_vertical_value',
            [
                'type'       => Controls_Manager::SLIDER,
                'show_label' => false,
                'size_units' => ['px', '%'],
                'range'      => [
                    'px' => [
                        'min'  => -1000,
                        'max'  => 1000,
                        'step' => 1,
                    ],
                    '%'  => [
                        'min' => -100,
                        'max' => 100,
                    ],
                ],
                'default'    => [
                    'unit' => '%',
                    'size' => 35,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .splide .splide__arrow--prev' => 'top: unset; bottom: unset; {{prev_vertical.value}}: {{SIZE}}{{UNIT}};',
                ]
            ]
        );
        $this->add_control(
            'prev_horizontal',
            [
                'label'       => esc_html__('Prev Horizontal', 'templatemela-core'),
                'type'        => Controls_Manager::CHOOSE,
                'label_block' => false,
                'options'     => [
                    'left'  => [
                        'title' => esc_html__('Left', 'templatemela-core'),
                        'icon'  => 'eicon-h-align-left',
                    ],
                    'right' => [
                        'title' => esc_html__('Right', 'templatemela-core'),
                        'icon'  => 'eicon-h-align-right',
                    ],
                ],
                'default'     => 'left'
            ]
        );
        $this->add_responsive_control(
            'prev_horizontal_value',
            [
                'type'       => Controls_Manager::SLIDER,
                'show_label' => false,
                'size_units' => ['px', 'em', '%'],
                'range'      => [
                    'px' => [
                        'min'  => -1000,
                        'max'  => 1000,
                        'step' => 1,
                    ],
                    '%'  => [
                        'min' => -100,
                        'max' => 100,
                    ],
                ],
                'default'    => [
                    'unit' => 'px',
                    'size' => 0,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .splide .splide__arrow--prev' => 'left: unset; right: unset; {{prev_horizontal.value}}: {{SIZE}}{{UNIT}};',
                ]
            ]
        );

        $this->end_controls_section();
		
        /// STYLE
        $this->start_controls_section(
			'section_style_category_icon',
			[
				'label' => esc_html__( 'Carousel Arrows', 'templatemela-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
                'conditions' => [
                    'relation' => 'and',
                    'terms'    => [
                        [
                            'name'     => 'enable_carousel',
                            'operator' => '==',
                            'value'    => 'yes',
                        ],
                        [
                            'name'     => 'navigation',
                            'operator' => '!==',
                            'value'    => 'none',
                        ],
                        [
                            'name'     => 'navigation',
                            'operator' => '!==',
                            'value'    => 'dots',
                        ],
                    ],
                ],
			]
		);

        $this->start_controls_tabs( 'icon_style', [
            'condition' => $condition,
		] );

        $this->start_controls_tab(
			'icon_normal',
			[
				'label' => esc_html__( 'Normal', 'templatemela-core' ),
			]
		);

        $this->add_control(
			'icon_color',
			[
				'label' => esc_html__( 'Color', 'templatemela-core' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .splide .splide__arrow' => 'color: {{VALUE}};',
				],
			]
		);

        $this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name' => 'icon_bg_color',
				'types' => [ 'classic', 'gradient' ],
				'exclude' => [ 'image' ],
				'selector' => '{{WRAPPER}} .splide .splide__arrow',
				'fields_options' => [
					'background' => [
						'default' => 'classic',
					],
					'color' => [
						'global' => [
							'default' => '',
						],
					],
				],
			]
		);

		$this->end_controls_tab();

        $this->start_controls_tab(
			'icon_hover',
			[
				'label' => esc_html__( 'Hover', 'templatemela-core' ),
			]
		);

        $this->add_control(
			'icon_hover_color',
			[
				'label' => esc_html__( 'Text Color', 'templatemela-core' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .splide .splide__arrow:hover, {{WRAPPER}} .splide .splide__arrow:focus' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name' => 'icon_bg_hover_color',
				'types' => [ 'classic', 'gradient' ],
				'exclude' => [ 'image' ],
				'selector' => '{{WRAPPER}} .splide .splide__arrow:hover, {{WRAPPER}} .splide .splide__arrow:focus',
				'fields_options' => [
					'background' => [
						'default' => 'classic',
					],
				],
			]
		);

        $this->add_control(
			'icon_hover_border_color',
			[
				'label' => esc_html__( 'Border Color', 'templatemela-core' ),
				'type' => Controls_Manager::COLOR,
				'condition' => [
					'border_border!' => '',
				],
				'selectors' => [
					'{{WRAPPER}} .splide .splide__arrow:hover, {{WRAPPER}} .splide .splide__arrow:focus' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

        $this->end_controls_tabs();

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name' => 'border',
				'selector' => '{{WRAPPER}} .splide .splide__arrow',
				'separator' => 'before',
			]
		);

		$this->add_responsive_control(
			'icon_radius',
			[
				'label' => esc_html__( 'Border Radius', 'templatemela-core' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em', 'rem', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .splide .splide__arrow' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name' => 'icon_box_shadow',
				'selector' => '{{WRAPPER}} .splide .splide__arrow',
			]
		);

		$this->end_controls_section();
		
    }

    protected function get_carousel_settings() {
        $settings = $this->get_settings_for_display();

        return array(
            'dimensions'       => $settings['dimensions'],
            'navigation'         => $settings['navigation'],
            'autoplayHoverPause' => $settings['pause_on_hover'] === 'yes' ? true : false,
            'autoplay'           => $settings['autoplay'] === 'yes' ? true : false,
            'autoplay_speed'     => $settings['autoplay_speed'],
            'items'              => $settings['column'],
            'items_laptop'       => isset($settings['column_laptop']) ? $settings['column_laptop'] : $settings['column'],
            'items_tablet'       => isset($settings['column_tablet']) ? $settings['column_tablet'] : $settings['column'],
            'items_mobile'       => isset($settings['column_mobile']) ? $settings['column_mobile'] : 1,
            'column_minimum'     => isset($settings['column_minimum']) ? $settings['column_minimum'] : 2,
            'loop'               => $settings['infinite'] === 'yes' ? true : false,
            'gap'              => $settings['gap'], 
            'gap_laptop'       => isset($settings['gap_laptop']) ? $settings['gap_laptop'] : 20,
            'gap_tablet'       => isset($settings['gap_tablet']) ? $settings['gap_tablet'] : 15,
            'gap_mobile'       => isset($settings['gap_mobile']) ? $settings['gap_mobile'] : 10,
            'categories_style'   => $settings['categories_style'],
        );
    }

    protected function render_carousel_template() {
        ?>
        var carousel_settings = {
        dimensions: settings.dimensions,
        navigation: settings.navigation,
        autoplayHoverPause: settings.pause_on_hover === 'yes' ? true : false,
        autoplay: settings.autoplay === 'yes' ? true : false,
        autoplay_speed: settings.autoplay_speed,
        items: settings.column,
        items_laptop: settings.column_laptop ? settings.column_laptop : settings.column,
        items_tablet: settings.column_tablet ? settings.column_tablet : settings.column,
        items_mobile: settings.column_mobile ? settings.column_mobile : 1,
        column_minimum: settings.column_minimum ? settings.column_minimum : 2,
        loop: settings.infinite === 'yes' ? true : false,
        gap: settings.gap,
        gap_laptop: settings.gap_laptop ? settings.gap_laptop : 20,
        gap_tablet: settings.gap_tablet ? settings.gap_tablet : 15,
        gap_mobile: settings.gap_mobile ? settings.gap_mobile : 10,
        categories_style: settings.categories_style,
        };
        <?php
    }
}

$widgets_manager->register(new TemplateMelaCore_Elementor_Product_Categories());
