<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use Base\Base_CSS;

/**
 * The main plugin class.
 *
 * @since 1.0.0
 * @package TemplateMela Core
 */

/**
 * The main templatemela-core class.
 */
class TemplateMelaCore_Plugin {

	/**
	* Plugin version, used for cache-busting of style and script file references.
	*
	* @since   1.0.0
	* @var  string
	*/
	const VERSION = TMCORE_VERSION;

	/**
	 * Instance of the class.
	 *
	 * @static
	 * @access protected
	 * @since 1.0.0
	 * @var object
	 */
	protected static $instance = null;

	/**
	 * Holds info if TemplateMela Settings is present.
	 *
	 * @static
	 * @access public
	 * @since 1.0.0
	 * @var bool
	 */
	public static $templatemela_settings_exists;

	private $suffix = '';

	/**
	 * Initialize the plugin by setting localization and loading public scripts
	 * and styles.
	 *
	 * @access private
	 * @since 1.0.0
	 */
	private function __construct() {
		$this->suffix = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? '' : '.min';

		add_action( 'plugins_loaded', [ $this, 'load_textdomain' ] );

		// Include Files Frontend and Backend
		$this->includes();
		
		if ( is_admin() || ( defined( 'WP_CLI' ) && WP_CLI ) ) {
			// Admin Only.
			$this->include_admin();
		} else {
			// Frontend Only.
			$this->include_frontend();
		}

		add_action( 'after_setup_theme', [ $this, 'set_templatemela_settings_exists' ] );

		// Load scripts & styles at Frontend and Backend.
		add_action( 'wp_enqueue_scripts', [ $this, 'scripts' ] );

		// Load scripts & styles at Frontend.
		if ( ! is_admin() ) {
			add_action( 'wp_enqueue_scripts', [ $this, 'scripts_frontend' ] );
		}

		add_filter( 'base_dynamic_css', array( $this, 'dynamic_css' ), 20 );

		// Register custom post-types and taxonomies.
		add_action( 'init', [ $this, 'register_post_types' ] );

		// User agent for news feed widget.
		add_action( 'wp_feed_options', [ $this, 'feed_user_agent' ], 10, 2 );

		// Register our admin widget.
		add_action( 'wp_dashboard_setup', [ $this, 'add_dashboard_widget' ], 100 );

		// Admin menu tweaks.
		add_action( 'admin_menu', [ $this, 'admin_menu' ] );

		// Plugin updater and activator
		add_action( 'after_setup_theme', [ $this, 'tmcore_updater_activator' ], 5 );

		// Load all shortcode elements.
		$this->init_shortcodes();

		// Register settings name.
		add_action( 'init', array( $this, 'load_api_settings' ) );

	}

	/**
	 * Plugin specific text-domain loader.
	 *
	 * @return void
	 */
	public function load_textdomain() {

		// Set filter for plugin's languages directory.
		$tmcore_lang_dir = TMCORE_PATH . 'languages/';
		$tmcore_lang_dir = apply_filters( 'tmcore_languages_directory', $tmcore_lang_dir );

		// Load the default language files.
		load_plugin_textdomain( 'templatemela-core', false, $tmcore_lang_dir );

	}

	/**
	 * Register settings
	 */
	public static function load_api_settings() {
		register_setting(
			'templatemela_core_module_config',
			'templatemela_core_module_config', // This is the data key for database.
			array(
				'type'              => 'string',
				'description'       => __( 'TemplateMela Module Config', 'templatemela-core' ),
				'sanitize_callback' => 'sanitize_text_field',
				'show_in_rest'      => true,
				'default'           => '',
			)
		);
	}

	/**
	 * Include files.
	 *
	 * @access public
	 * @since 1.0.0
	 * @return void
	 */
	public function includes() {

		// Load Settings Class.
		require_once TMCORE_PATH . 'includes/class-templatemela-settings.php';
		TemplateMelaCore_Settings::load_api_settings();
		
		// Load TemplateMela Functions.
		require_once TMCORE_PATH . 'includes/templatemela-functions.php';

		// Load widget classes.
		require_once( TMCORE_PATH . 'includes/class-templatemela-widget.php' );
		require_once( TMCORE_PATH . 'includes/widgets/class-templatemela-widgets.php' );

		if ( false == templatemela_check_required_theme() ) {
			return;
		}
		// Load addons.
		require_once TMCORE_PATH . 'includes/addons/elements/post-select-rest-controller.php';
		add_action( 'rest_api_init', array( $this, 'register_api_endpoints' ) );
		$enabled = json_decode( get_option( 'templatemela_core_module_config' ), true );

		if ( empty( $enabled ) ){
			$module_config = [ 'header_addons' => true, 'conditional_headers' => true, 'mega_menu' => true, 'scripts' => true, 'elements' => true, 'woocommerce_addons' => true, 'infinite' => true, 'dark_mode' => true, 'localgravatars' => true, 'archive_meta' => true ];	
			update_option("templatemela_core_module_config", json_encode($module_config));
		}

		if ( isset( $enabled ) && isset( $enabled['conditional_headers'] ) && true === $enabled['conditional_headers'] ) {
			require_once TMCORE_PATH . 'includes/addons/conditional-headers.php';
		}
		if ( isset( $enabled ) && isset( $enabled['elements'] ) && true === $enabled['elements'] ) {
			require_once TMCORE_PATH . 'includes/addons/elements/duplicate-elements.php';
			require_once TMCORE_PATH . 'includes/addons/elements/elements-init.php';
		}
		if ( isset( $enabled ) && isset( $enabled['header_addons'] ) && true === $enabled['header_addons'] ) {
			require_once TMCORE_PATH . 'includes/addons/header-addons.php';
		}
		if ( isset( $enabled ) && isset( $enabled['mega_menu'] ) && true === $enabled['mega_menu'] ) {
			require_once TMCORE_PATH . 'includes/addons/mega-menu/mega-menu.php';
		}
		if ( class_exists( 'woocommerce' ) && isset( $enabled ) && isset( $enabled['woocommerce_addons'] ) && true === $enabled['woocommerce_addons'] ) {
			require_once TMCORE_PATH . 'includes/addons/woocommerce-addons.php';
		}
		if ( isset( $enabled ) && isset( $enabled['scripts'] ) && true === $enabled['scripts'] ) {
			require_once TMCORE_PATH . 'includes/addons/scripts-addon.php';
		}
		if ( isset( $enabled ) && isset( $enabled['infinite'] ) && true === $enabled['infinite'] ) {
			require_once TMCORE_PATH . 'includes/addons/infinite-scroll.php';
		}
		if ( isset( $enabled ) && isset( $enabled['localgravatars'] ) && true === $enabled['localgravatars'] ) {
			require_once TMCORE_PATH . 'includes/addons/local-gravatars.php';
		}
		if ( isset( $enabled ) && isset( $enabled['archive_meta'] ) && true === $enabled['archive_meta'] ) {
			require_once TMCORE_PATH . 'includes/addons/archive-meta.php';
		}
		if ( isset( $enabled ) && isset( $enabled['dark_mode'] ) && true === $enabled['dark_mode'] ) {
			require_once TMCORE_PATH . 'includes/addons/dark-mode.php';
		}
		
		// More Addons
		require_once TMCORE_PATH . 'includes/addons/additional.php';

		// WooCommerce
		if ( class_exists( 'WooCommerce' ) ) {

			// Useful wooocommerce classes.
			require_once TMCORE_WOO_PATH . 'classes/base-woo-extras-plugin-check.php';
			require_once TMCORE_WOO_PATH . 'classes/class-base-image-processing.php';
			require_once TMCORE_WOO_PATH . 'classes/class-base-woo-get-image.php';
			require_once TMCORE_WOO_PATH . 'classes/custom_functions.php';
			require_once TMCORE_WOO_PATH . 'classes/class-base-woo-duplicate-post.php';
			require_once TMCORE_WOO_PATH . 'inc/class-base-woo-options.php';
			require_once TMCORE_WOO_PATH . 'inc/class-base-woo-extras-settings.php';
			require_once TMCORE_WOO_PATH . 'classes/cmb/init.php';
			require_once TMCORE_WOO_PATH . 'classes/class-base-woo-css.php';
			require_once TMCORE_WOO_PATH . 'classes/class-base-woo-google-fonts.php';
			require_once TMCORE_WOO_PATH . 'classes/cmb2-conditionals/cmb2-conditionals.php';
			require_once TMCORE_WOO_PATH . 'classes/cmb2_select2/cmb_select2.php';
			// woocommerce functions
			require_once TMCORE_WOO_PATH . 'woocommerce-functions.php';
			require_once TMCORE_WOO_PATH . 'class-woocommerce.php';
			require_once TMCORE_WOO_PATH . 'class-woocommerce-adjacent-products.php';
			require_once TMCORE_WOO_PATH . 'product-query.php';
			// Taxonomy Meta
			require_once TMCORE_WOO_PATH . 'classes/taxonomy-meta-class.php';
			// For WPC Plugins
			if ( function_exists('woosw_init') || function_exists('woosc_init') || function_exists('woosq_init') ) {
				require_once TMCORE_WOO_PATH . 'class-woocommerce-clever.php';
			}
			// For Dokan Plugin
			if ( function_exists('dokan') ) {
				require_once TMCORE_WOO_PATH . 'class-woocommerce-dokan.php';			
			}

			if ( base_woo_extras_is_woo_active() ) {
				$shopkit_settings = get_option( 'bt_woo_extras' );
				if ( ! is_array( $shopkit_settings ) ) {
					$shopkit_settings = json_decode( $shopkit_settings, true );
				}
				require_once TMCORE_WOO_PATH . 'lib/variations/bt-variations-price.php';
				if ( isset( $shopkit_settings['snackbar_notices'] ) && $shopkit_settings['snackbar_notices'] ) {
					require_once TMCORE_WOO_PATH . 'lib/snackbar/class-snackbar-notices.php';
				}
				if ( isset( $shopkit_settings['variation_swatches'] ) && $shopkit_settings['variation_swatches'] ) {
					require_once TMCORE_WOO_PATH . 'lib/swatches/bt-variations-swatches.php';
				}
				if ( isset( $shopkit_settings['product_gallery'] ) && $shopkit_settings['product_gallery'] ) {
					require_once TMCORE_WOO_PATH . 'lib/gallery/class-product-gallery.php';
					require_once TMCORE_WOO_PATH . 'lib/variation-gallery/base-variation-gallery.php';
				}
				if ( isset( $shopkit_settings['size_charts'] ) && $shopkit_settings['size_charts'] ) {
					require_once TMCORE_WOO_PATH . 'lib/sizechart/bt-size-chart.php';
				}
				if ( isset( $shopkit_settings['bt_add_to_cart_text'] ) && $shopkit_settings['bt_add_to_cart_text'] ) {
					require_once TMCORE_WOO_PATH . 'lib/add_to_cart_text/bt-add-to-cart-text.php';
				}
				if ( isset( $shopkit_settings['bt_reviews'] ) && $shopkit_settings['bt_reviews'] ) {
					require_once TMCORE_WOO_PATH . 'lib/reviews/reviews.php';
				}
				if ( isset( $shopkit_settings['bt_cart_notice'] ) && $shopkit_settings['bt_cart_notice'] ) {
					require_once TMCORE_WOO_PATH . 'lib/cartnotice/bt-cart-notice.php';
				}
				if ( isset( $shopkit_settings['bt_extra_cat'] ) && $shopkit_settings['bt_extra_cat'] ) {
					require_once TMCORE_WOO_PATH . 'lib/extracatdesc/bt-extra-cat-desc.php';
				}
				if ( isset( $shopkit_settings['bt_checkout_editor'] ) && $shopkit_settings['bt_checkout_editor'] ) {
					require_once TMCORE_WOO_PATH . 'lib/checkout_editor/bt-checkout-editor.php';
				}
				if ( isset( $shopkit_settings['bt_affiliate_options'] ) && $shopkit_settings['bt_affiliate_options'] ) {
					require_once TMCORE_WOO_PATH . 'lib/affiliate/bt-affiliate-options.php';
				}
				if ( isset( $shopkit_settings['bt_product_brands_options'] ) && $shopkit_settings['bt_product_brands_options'] ) {
					require_once TMCORE_WOO_PATH . 'lib/brands/class-bt-extra-brands.php';
				}
				if ( isset( $shopkit_settings['bt_coupon_modal_checkout'] ) && $shopkit_settings['bt_coupon_modal_checkout'] ) {
					require_once TMCORE_WOO_PATH . 'lib/checkout_coupon/bt-checkout-coupon.php';
				}
				if ( isset( $shopkit_settings['bt_global_tabs'] ) && $shopkit_settings['bt_global_tabs'] ) {
					require_once TMCORE_WOO_PATH . 'lib/tabs/class-base-global-tabs.php';
				}
			}
			
		}
		
		// Elementor
		if ( did_action( 'elementor/loaded' ) ) {
			require_once TMCORE_PATH . 'includes/elementor/elementor-functions.php';
			require_once TMCORE_PATH . 'includes/elementor/class-elementor.php';
		}
		
	}

	/**
	 * On Load
	 */
	public function include_admin() {
		// Include Admin Files
		require_once TMCORE_PATH . 'admin/class-theme-admin.php';
	}

	/**
	 * On Load
	 */
	public function include_frontend() {
		// Load popup classes.
		require_once( TMCORE_PATH . 'includes/class-templatemela-popup.php' );
	}

	/**
	 * Sets the TemplateMela Library constant.
	 *
	 * @access public
	 * @since 1.0.0
	 * @return void
	 */
	public function set_templatemela_settings_exists() {
		self::$templatemela_settings_exists = class_exists( 'TemplateMelaCore_Settings' );
	}

	/**
	 * Return an instance of this class.
	 *
	 * @static
	 * @access public
	 * @since 1.0.0
	 * @return object  A single instance of the class.
	 */
	public static function get_instance() {

		// If the single instance hasn't been set yet, set it now.
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;

	}

	/**
	 * Returns a cached query.
	 * If the query is not cached then it caches it and returns the result.
	 *
	 * @static
	 * @access public
	 * @param string|array $args Same as in WP_Query.
	 * @return object
	 */
	public static function tmcore_cached_query( $args ) {

		$query_id = md5( maybe_serialize( $args ) );
		$query    = wp_cache_get( $query_id, 'tmcore' );
		if ( false === $query ) {
			$query = new WP_Query( $args );
			wp_cache_set( $query_id, $query, 'tmcore' );
		}
		return $query;

	}

	/**
	 * Enqueues scripts at Frontend and Backend.
	 *
	 * @access public
	 */
	public function scripts() {}
	
	/**
	 * Enqueues scripts at Frontend.
	 *
	 * @access public
	 */
	public function scripts_frontend() {
		
		wp_enqueue_script( 'tmcore-helpers', TMCORE_URL . "assets/js/helpers{$this->suffix}.js", array( 'jquery' ), TMCORE_VERSION, false );
		wp_enqueue_style( 'perfect-scrollbar', TMCORE_URL . 'assets/js/perfect-scrollbar/perfect-scrollbar.min.css', array(), TMCORE_VERSION );
		wp_enqueue_script( 'perfect-scrollbar', TMCORE_URL . 'assets/js/perfect-scrollbar/perfect-scrollbar.min.js', array( 'jquery' ), TMCORE_VERSION, true );
		wp_enqueue_script( 'tmcore-countdown', TMCORE_URL . "assets/js/countdown{$this->suffix}.js", array('jquery'), TMCORE_VERSION, true );
		wp_enqueue_style( 'tmcore-frontend', TMCORE_URL . "assets/css/frontend{$this->suffix}.css", [], TMCORE_VERSION );
		wp_enqueue_script( 'tmcore-frontend-js', TMCORE_URL . "assets/js/frontend{$this->suffix}.js", array( 'jquery' ), TMCORE_VERSION, true );
		
		$tm_settings = get_option( 'bt_woo_extras' );
		if ( ! is_array( $tm_settings ) ) {
			$tm_settings = json_decode( $tm_settings, true );
		}

		if ( isset( $tm_settings['loading_screen'] ) && $tm_settings['loading_screen'] ) {
			wp_enqueue_script( 'tmcore-loading-screen', TMCORE_URL . "assets/js/loading-screen{$this->suffix}.js", array('jquery'), TMCORE_VERSION, false );
			wp_localize_script( 'tmcore-loading-screen', 'tmcore_loading_screen_setting', array('loading_image' => TMCORE_URL . 'assets/images/loading_screen.svg' ) );
		}
		
		if ( isset( $tm_settings['skeleton_shimmer'] ) && $tm_settings['skeleton_shimmer'] ) {
			wp_enqueue_style( 'tmcore-skeleton', TMCORE_URL . "assets/css/skeleton{$this->suffix}.css", [], TMCORE_VERSION );

		}
	}


	/**
	 * Register custom post types.
	 *
	 * @access public
	 * @since 1.0.0
	 */
	public function register_post_types() {}

	/**
	 * Elastic Slider admin menu.
	 *
	 * @access public
	 */
	public function admin_menu() {}

	/**
	 * Adds the news dashboard widget.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return void
	 */
	public function add_dashboard_widget() {

		// Create the widget.
		wp_add_dashboard_widget( 'templatemela_news', apply_filters( 'tmcore_dashboard_widget_title', esc_attr__( 'TemplateMela News and Updates', 'TemplateMela' ) ), [ $this, 'display_news_dashboard_widget' ] );

		// Make sure our widget is on top off all others.
		global $wp_meta_boxes;

		// Get the regular dashboard widgets array.
		$normal_dashboard = $wp_meta_boxes['dashboard']['normal']['core'];

		$templatemela_widget_backup = [];
		if ( isset( $normal_dashboard['templatemela_news'] ) ) {
			// Backup and delete our new dashboard widget from the end of the array.
			$templatemela_widget_backup = [
				'templatemela_news' => $normal_dashboard['templatemela_news'],
			];
			unset( $normal_dashboard['templatemela_news'] );
		}

		// Merge the two arrays together so our widget is at the beginning.
		$sorted_dashboard = array_merge( $templatemela_widget_backup, $normal_dashboard );

		// Save the sorted array back into the original metaboxes.
		$wp_meta_boxes['dashboard']['normal']['core'] = $sorted_dashboard; // phpcs:ignore WordPress.WP.GlobalVariablesOverride
	}

	/**
	 * Renders the news dashboard widget.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return void
	 */
	public function display_news_dashboard_widget() {

		// Create two feeds, the first being just a leading article with data and summary, the second being a normal news feed.
		$feeds = [
			'news' => [
				'link'         => 'https://templatemela.com/blog/',
				'url'          => 'https://templatemela.com/blog/category/wordpress/feed/',
				'title'        => esc_attr__( 'TemplateMela News and Updates', 'templatemela-core' ),
				'items'        => 4,
				'show_summary' => 0,
				'show_author'  => 0,
				'show_date'    => 0,
			],
		];
		?>
		<div class="templatemela-news-wrapper">
			<div class="templatemela-news-header">
				<div class="templatemela-news-logo">
					<span class="templatemela-news-image-wrapper">
						<img src="<?php echo esc_url( TMCORE_URL . 'admin/assets/images/templatemela-logo.png' ); ?>" width="135" height="20" alt="<?php esc_attr_e( 'TemplateMela Logo', 'templatemela-core' ); ?>">
					</span>
				</div>
				<div class="templatemela-theme-info">
					<?php $version = defined( 'TMCORE_VERSION' ) ? 'v' . TMCORE_VERSION : ''; ?>
					<?php $name = defined( 'TMCORE_NAME' ) ? TMCORE_NAME : ''; ?>
					<span class="templatemela-core-name"><strong><?php echo esc_html( apply_filters( 'tmcore_name', $name ) ); ?></strong></span>
					<span class="templatemela-news-version"><?php echo esc_html( apply_filters( 'tmcore_db_version', $version ) ); ?></span>
				</div>
			</div>
			<?php wp_dashboard_primary_output( 'templatemela_news', $feeds ); ?>

			<div class="templatemela-news-footer">
				<ul>
					<li class="templatemela-news-footer-blog>"><a href="<?php echo esc_url( 'https://templatemela.com/blog/' ); ?>" target="_blank"><?php esc_html_e( 'Blog', 'templatemela-core' ); ?> <span class="screen-reader-text"><?php esc_html_e( '(opens in a new window)', 'templatemela-core' ); ?></span><span aria-hidden="true" class="dashicons dashicons-external"></span></a></li>
					<li class="templatemela-news-footer-docs>"><a href="<?php echo esc_url( 'https://templatemela.com/docs/' ); ?>" target="_blank"><?php esc_html_e( 'Docs', 'templatemela-core' ); ?> <span class="screen-reader-text"><?php esc_html_e( '(opens in a new window)', 'templatemela-core' ); ?></span><span aria-hidden="true" class="dashicons dashicons-external"></span></a></li>
					<li class="templatemela-news-footer-ticket>"><a href="<?php echo esc_url( 'https://templatemela.com/support/' ); ?>" target="_blank"><?php esc_html_e( 'Ticket', 'templatemela-core' ); ?> <span class="screen-reader-text"><?php esc_html_e( '(opens in a new window)', 'templatemela-core' ); ?></span><span aria-hidden="true" class="dashicons dashicons-external"></span></a></li>
				</ul>
			</div>
		</div>
		<?php
	}

	/**
	 * Changes the user agent for the TemplateMela news widget.
	 *
	 * @since 1.0.0
	 * @access public
	 * @param  object $feed  SimplePie feed object, passed by reference.
	 * @param  mixed  $url   URL of feed to retrieve. If an array of URLs, the feeds are merged.
	 * @return void
	 */
	public function feed_user_agent( $feed, $url ) {

		if ( 'https://templatemela.com/blog/category/wordpress/feed/' === $url ) {
			$feed->set_useragent( 'TemplateMela RSS Feed' );
		}
	}

	/**
	 * Plugin updater and activator.
	 * 
	 * @since 1.0.0
	 * @access public
	 */
	public function tmcore_updater_activator() {
		// Load only in administration panel
		if( is_admin() ) {
			// Load the Theme Plugins update checker.
			require_once TMCORE_PATH . 'admin/base-update-checker/base-update-checker.php';
			// Load the update API.
			require_once TMCORE_PATH . 'admin/base-activation/class-base-api-manager.php';
			
		}
	}

	/**
	 * Find and include all shortcodes within shortcodes folder.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public function init_shortcodes() {
			$filenames = glob( TMCORE_PATH . 'includes/shortcodes/*.php', GLOB_NOSORT );
			foreach ( $filenames as $filename ) {
				$info = pathinfo( $filename );
				require_once wp_normalize_path( $filename );
			}
	}
	
	/**
	 * Setup the post select API endpoint.
	 *
	 * @return void
	 */
	public function register_api_endpoints() {
		$controller = new BasePro\Post_Select_Controller();
		$controller->register_routes();
	}

	/**
	 * Generates the dynamic css based on customizer options.
	 *
	 * @param string $css any custom css.
	 * @return string
	 */
	public function dynamic_css( $css ) {
		$generated_css = $this->generate_core_css();
		if ( ! empty( $generated_css ) ) {
			$css .= "\n/* TemplateMela Core CSS */\n" . $generated_css;
		}
		return $css;
	}

	/**
	 * Generates the dynamic css based on page options.
	 *
	 * @return string
	 */
	public function generate_core_css() {
		$css                    = new Base_CSS();
		$media_query            = array();
		$media_query['mobile']  = apply_filters( 'base_mobile_media_query', '(max-width: 767px)' );
		$media_query['tablet']  = apply_filters( 'base_tablet_media_query', '(max-width: 1024px)' );
		$media_query['desktop'] = apply_filters( 'base_tablet_media_query', '(min-width: 1025px)' );

		$tm_settings = get_option( 'bt_woo_extras' );
		if ( ! is_array( $tm_settings ) ) {
			$tm_settings = json_decode( $tm_settings, true );
		}

        if ( isset( $tm_settings['skeleton_shimmer_color'] ) && !empty($tm_settings['skeleton_shimmer_color']) ) {
            $css->set_selector( ':root' );
            $css->add_property( '--skeleton-gradient-color', $tm_settings['skeleton_shimmer_color'] );
        }
        return $css->css_output();
	}


}
