<?php
/**
 * The main plugin class.
 *
 * @since 1.0
 * @package TemplateMela Theme
 */

use function Base\webapp;
use Base\Base_CSS;

/**
 * The main templatemela-theme class.
 */
class TemplateMela_Plugin {

	/**
	* Plugin version, used for cache-busting of style and script file references.
	*
	* @since   1.0.0
	* @var  string
	*/
	const VERSION = TMTHEME_VERSION;

	/**
	 * Instance of the class.
	 *
	 * @static
	 * @access protected
	 * @since 1.0.0
	 * @var object
	 */
	protected static $instance = null;

	/**
	 * JS folder URL.
	 *
	 * @static
	 * @access public
	 * @since 1.0.0
	 * @var string
	 */
	public static $assets_folder_url;

	/**
	 * JS folder path.
	 *
	 * @static
	 * @access public
	 * @since 1.0.0
	 * @var string
	 */
	public static $assets_folder_path;

	/**
	 * Holds info if FL is present.
	 *
	 * @static
	 * @access public
	 * @since 1.0.0
	 * @var bool
	 */
	public static $templatemela_settings_exists;

	/**
	 * Holds theme array sections.
	 *
	 * @var the theme settings sections.
	 */
	private $default_options = array();
	private $header_addons = array();
	private $more_addons = array();
	private $woo_addons = array();
	private $dark_mode_addons = array();

	/**
	 * Holds default palette values
	 *
	 * @var values of the theme settings.
	 */
	protected static $default_palette = null;

	/**
	 * Initialize the plugin by setting localization and loading public scripts
	 * and styles.
	 *
	 * @access private
	 * @since 1.0.0
	 */
	private function __construct() {

		if ( isset( $_REQUEST['select_layout']) ){
			delete_option( 'base_global_palette' );
			update_option( 'harvest_selected_layout', sanitize_title( strtolower( $_REQUEST['select_layout'] ) ) );
		}

		$path                 = ( defined( 'TMTHEME_DEV_MODE' ) && true === TMTHEME_DEV_MODE ) ? '' : '/min';
		self::$assets_folder_url  = TMTHEME_URL . 'assets' . $path;
		self::$assets_folder_path = TMTHEME_PATH . '/assets' . $path;

		// Always load text domain first.
		add_action( 'plugins_loaded', [ $this, 'load_textdomain' ] );

		// Include Files Frontend and Backend.
		$this->includes();
		
		if ( is_admin() || ( defined( 'WP_CLI' ) && WP_CLI ) ) {
			// Admin Only.
			$this->include_admin();
		} else {
			// Frontend Only.
			$this->include_frontend();
		}

		// Check if TemplateMela Settings Class is exists.
		add_action( 'after_setup_theme', [ $this, 'set_templatemela_settings_exists' ] );

		// Plugin updater and activator
		add_action( 'after_setup_theme', [ $this, 'templatemela_updater_activator' ], 5 );

		// Set Default colors of this theme.
		add_filter( 'base_global_palette_defaults', [ $this, 'add_color_option_defaults' ], 10 );

		// Set Default Options of Customizer for this theme.
		add_filter( 'base_theme_options_defaults', [ $this, 'add_option_defaults' ], 10 );

		add_filter( 'base_dynamic_css', [ $this, 'dynamic_css' ], 30 );

		// Override woocommerce plugin files
		add_filter( 'woocommerce_locate_template', [ $this, 'override_plugin_files' ], 10, 3 ); 

		// Allow SVG to upload
		add_filter( 'upload_mimes', [ $this, 'upload_svg_icons' ]);

		// Change number of products(columns) per row(line)
		add_filter( 'loop_shop_columns', [ $this, 'tmcore_loop_columns' ] );
		
		// Change Theme Layout.
		add_action( 'base_theme_dash_section_bottom', [ $this, 'change_layout_options' ] );

		// Register custom post-types and taxonomies.
		//add_action( 'init', [ $this, 'register_post_types' ] );

		// Admin menu tweaks.
		//add_action( 'admin_menu', [ $this, 'admin_menu' ] );

		// Init Widgets.
		//add_action( 'widgets_init', [ $this, 'widget_init' ] );

		// Load scripts & styles at Frontend and Backend.
		//add_action( 'wp_enqueue_scripts', [ $this, 'scripts' ] );

		// Load scripts & styles at Frontend.
		/* if ( ! is_admin() ) {
			add_action( 'wp_enqueue_scripts', [ $this, 'scripts_frontend' ] );
		} */

		// Load all shortcode elements.
		$this->init_shortcodes();



	}

	/**
	 * Plugin specific text-domain loader.
	 *
	 * @return void
	 */
	public function load_textdomain() {

		// Set filter for plugin's languages directory.
		$tmtheme_lang_dir = TMTHEME_PATH . '/languages/';
		$tmtheme_lang_dir = apply_filters( 'tmtheme_languages_directory', $tmtheme_lang_dir );

		// Load the default language files.
		load_plugin_textdomain( 'templatemela-plugin-harvest', false, $tmtheme_lang_dir );

	}

	/**
	 * Include files.
	 *
	 * @access public
	 * @since 1.0.0
	 * @return void
	 */
	public function includes() {
		
		// Load widget classes.
		$filenames = glob( TMTHEME_PATH . '/includes/widgets/*.php', GLOB_NOSORT );
		foreach ( $filenames as $filename ) {
			require_once wp_normalize_path( $filename );
		}

		$selected_layout = get_option( 'harvest_selected_layout' );

		if( ! empty( $selected_layout ) &&  'default' != $selected_layout ) {
			require_once TMTHEME_PATH . '/layouts/'.$selected_layout.'/functions.php';
		} else {
			require_once TMTHEME_PATH . '/layouts/default/functions.php';
		}
	}

	/**
	 * On Load
	 */
	public function include_admin() {
		
		// Include Admin Files
		require_once TMTHEME_PATH . '/admin/class-theme-admin.php';
	}

	/**
	 * On Load
	 */
	public function include_frontend() {}

	/**
	 * Sets the TemplateMela Library constant.
	 *
	 * @access public
	 * @since 1.0.0
	 * @return void
	 */
	public function set_templatemela_settings_exists() {
		self::$templatemela_settings_exists = class_exists( 'TemplateMelaCore_Settings' );
	}

	/**
	 * Return an instance of this class.
	 *
	 * @static
	 * @access public
	 * @since 1.0.0
	 * @return object  A single instance of the class.
	 */
	public static function get_instance() {

		// If the single instance hasn't been set yet, set it now.
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;

	}

	/**
	 * Returns a cached query.
	 * If the query is not cached then it caches it and returns the result.
	 *
	 * @static
	 * @access public
	 * @param string|array $args Same as in WP_Query.
	 * @return object
	 */
	public static function templatemela_plugin_cached_query( $args ) {

		$query_id = md5( maybe_serialize( $args ) );
		$query    = wp_cache_get( $query_id, 'tmtheme' );
		if ( false === $query ) {
			$query = new WP_Query( $args );
			wp_cache_set( $query_id, $query, 'tmtheme' );
		}
		return $query;

	}

	/**
	 * Enqueues scripts at Frontend and Backend.
	 *
	 * @access public
	 */
	public function scripts() {}
	
	/**
	 * Enqueues scripts at Frontend.
	 *
	 * @access public
	 */
	public function scripts_frontend() {
		wp_enqueue_style( 'templatemela-theme-frontend', TMTHEME_URL . 'assets/css/frontend.css', [], TMTHEME_VERSION );
	}


	/**
	 * Register custom post types.
	 *
	 * @access public
	 * @since 1.0.0
	 */
	public function register_post_types() {}

	/**
	 * Elastic Slider admin menu.
	 *
	 * @access public
	 */
	public function admin_menu() {}


	/**
	 * Register widgets.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return void
	 */
	public function widget_init() {
		register_widget( 'TemplateMela_Widget_Example' );
	}


	/**
	 * Plugin updater and activator.
	 * 
	 * @since 1.0.0
	 * @access public
	 */
	public function templatemela_updater_activator() {
		// Load only in administration panel
		if( is_admin() ) {
			// Load the Theme Plugins update checker.
			require_once TMTHEME_PATH . '/admin/base-update-checker/base-update-checker.php';			
		}
	}

	/**
	 * Find and include all shortcodes within shortcodes folder.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public function init_shortcodes() {
			$filenames = glob( TMTHEME_PATH . '/includes/shortcodes/*.php', GLOB_NOSORT );
			foreach ( $filenames as $filename ) {
				$info = pathinfo( $filename );
				require_once wp_normalize_path( $filename );
			}
	}

	/**
	 * set theme Default color.
	 */
	public function add_color_option_defaults( $defaults ) {
		if ( is_null( self::$default_palette ) ) {

			$selected_layout = get_option( 'harvest_selected_layout' );

			if( ! empty( $selected_layout ) &&  'default' != $selected_layout ) {
				require_once TMTHEME_PATH . '/layouts/'.$selected_layout.'/palette.php';
			} else {
				require_once TMTHEME_PATH . '/layouts/default/palette.php';
			}
			self::$default_palette = $color_palette;
		}
		return self::$default_palette;
	}

	public function add_option_defaults( $defaults ) {

		$selected_layout = get_option( 'harvest_selected_layout' );

		if( ! empty( $selected_layout ) &&  'default' != $selected_layout ) {
			require_once TMTHEME_PATH . '/layouts/'.$selected_layout.'/default-options.php';
		} else {
			require_once TMTHEME_PATH . '/layouts/default/default-options.php';
		}

		$defaults = array_merge(
			$defaults,
			isset($default_options) ? $default_options : array(),
			isset($header_addons) ? $header_addons : array(),
			isset($more_addons) ? $more_addons : array(),
			isset($woo_addons) ? $woo_addons : array(),
			isset($dark_mode_addons) ? $dark_mode_addons : array(),
		);
		return $defaults;
	}
	
	public function dynamic_css( $css ) {
		$generated_css = $this->generate_theme_css();
		if ( ! empty( $generated_css ) ) {
		$css .= "\n/* TemplateMela CSS */\n" . $generated_css;
		}
		return $css;
	}

	public function generate_theme_css () {

		$css                    = new Base_CSS();
		$media_query            = array();
		$media_query['mobile']  = apply_filters( 'base_mobile_media_query', '(max-width: 767px)' );
		$media_query['tablet']  = apply_filters( 'base_tablet_media_query', '(max-width: 1024px)' );
		$media_query['desktop'] = apply_filters( 'base_tablet_media_query', '(min-width: 1025px)' );
		// Account CSS.

		$css->set_selector( '.primary-sidebar.widget-area .widget-title, .widget_block h2,.widget_block .widgettitle,.widget_block .widgettitle,.primary-sidebar h2' );
		$css->render_font( webapp()->option( 'sidebar_widget_title' ), $css );
		return $css->css_output();
	}

	/**
	 * Filter the woocommerce template path to use in this plugin instead of the one in WooCommerce.
	 *
	 * @param string $template      Default template file path.
	 * @param string $template_name Template file slug.
	 * @param string $template_path Template file name.
	 *
	 * @return string The new Template file path.
	 */
	// Override woocommerce plugin files
	function override_plugin_files( $template, $template_name, $template_path ) {

		$template_directory = trailingslashit( TMTHEME_PATH ) . 'includes/woocommerce/';
		$path = $template_directory . $template_name;
		
		return file_exists( $path ) ? $path : $template;
	}
	

	// Upload SVG icons
	function upload_svg_icons($file_types){
		$new_filetypes = array();
		$new_filetypes['svg'] = 'image/svg+xml';
		$file_types = array_merge($file_types, $new_filetypes );
		return $file_types;
	}

	/**
	 * Change number of products(columns) per row(line)
	 *
	 * @access public
	 */
	public function tmcore_loop_columns() {

		//global $shopkit_settings;
		$shopkit_settings = get_option( 'bt_woo_extras' );
		if ( ! is_array( $shopkit_settings ) ) {
			$shopkit_settings = json_decode( $shopkit_settings, true );
		}
		
		$product_per_row = 4;
		if ( isset( $shopkit_settings['bt_products_per_row'] ) && $shopkit_settings['bt_products_per_row'] ) {
			$product_per_row = $shopkit_settings['bt_products_per_row'];    
		}
		return $product_per_row; // 4 products per row
		//return \TemplateMelaCore_Settings::get_setting_option( 'tmcore_shop_items_per_row', 4 ); // 4 products per row
	}
	
	public function change_layout_options() {

		$selected_layout = get_option( 'harvest_selected_layout' );

		if( empty( $selected_layout ) ||  'default' == $selected_layout ) {
			$selected_layout = 'default';
		}

		echo '<br/><br/>
		<div class="dashboard-section tab-section components-panel">
			<div class="components-panel__body is-opened">
				<div class="tmc-dashboard-head">
					<h2 class="section-header">Current Layout : ' . ucfirst( $selected_layout ) . '</h2>
				</div>
				<div class="tmc-dashboard-contents-3col">
					<form method="post" action="' . admin_url( sprintf( basename( $_SERVER['REQUEST_URI'] ) ) ) . '" id="change_layout">
					<div class="dashboard-customizer-settings">
						<h4 class="section-header">Select below to change theme layout</h4>
						<div class="two-col-grid">';
						$dirs = glob( TMTHEME_PATH . '/layouts/*', GLOB_ONLYDIR ); $class = '';
						foreach ( $dirs as $dir ) {
							if ( strtolower( $selected_layout ) == basename( $dir ) ) { $sel = 'checked="checked"'; $class ="selected";} else { $sel = ''; }
							echo '<div class="link-item '. $class.' ">';
							echo '<input type="radio" id="' . basename( $dir ) . '" name="select_layout" value="' . basename( $dir ) . '" '. $sel.' style="visibility:hidden" >';
							echo '<label for="' . basename( $dir ) . '"><img style="max-width:240px" width="auto" src="' . TMTHEME_URL . 'layouts/' . basename( $dir ) . '/screenshot.png" alt="' . basename( $dir ) . '" />';
							echo '<div>' . ucfirst( basename( $dir ) ) . '</div>';
							if ( strtolower( $selected_layout ) == basename( $dir ) ) {
								echo '<div class="link-item-foot" style="margin-top:15px"><a class="button-primary">Selected</a></div>';
							} else { 
								echo '<div class="link-item-foot" style="margin-top:15px"><a class="button-secondary">Select</a></div>';
							}
							echo '</label></div>';
						}
						echo '
						</div>
					</div>
					</form>
				</div>
			</div>
		</div>
		<script>jQuery(document).ready(function($) { $(".link-item input[type=radio]").on("change", function() { $(this).closest("form").submit(); }); });</script>';

	}	

}
