<?php

if (!defined('ABSPATH')) {
    exit;
}

use function Base\webapp;

if ( !class_exists('TemplateMelaCore_WooCommerce') ) :

    /**
     * The TemplateMela WooCommerce Integration class
     */
    class TemplateMelaCore_WooCommerce {

        public $list_shortcodes;

        private $suffix = '';

        private $shopkit_settings = '';

        /**
         * Setup class.
         *
         * @since 1.0
         */
        public function __construct() {
            $this->suffix = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? '' : '.min';
            
            $this->shopkit_settings = get_option( 'bt_woo_extras' );
            if ( ! is_array( $this->shopkit_settings ) ) {
                $this->shopkit_settings = json_decode( $this->shopkit_settings, true );
            }
            
            // Depricated, back compatability for theme-plugin
            global $shopkit_settings;
            $shopkit_settings = $this->shopkit_settings;
            
            $product_per_row = 4;
            if ( isset( $this->shopkit_settings['bt_products_per_row'] ) && !empty($this->shopkit_settings['bt_products_per_row']) ) {
                $product_per_row = (int)$this->shopkit_settings['bt_products_per_row'];    
            }
            \TemplateMelaCore_Settings::update_setting_option( 'tmcore_shop_items_per_row', $product_per_row );
            // will be removed in next major version

            $this->list_shortcodes = array(
                'recent_products',
                'sale_products',
                'best_selling_products',
                'top_rated_products',
                'featured_products',
                'related_products',
                'product_category',
                'products',
            );

            $this->init_shortcodes();

            add_action('after_setup_theme', array($this, 'woocommerce_setup'), 99);
            add_filter('body_class', array($this, 'woocommerce_body_class'));
            add_action('widgets_init', array($this, 'widgets_init'));
            add_filter('tmcore_theme_sidebar', array($this, 'set_sidebar'), 20);
            add_action('wp_enqueue_scripts', array( $this, 'action_enqueue_scripts' ), 999);
                        
            // Trim zeros in price decimals
            add_filter( 'woocommerce_price_trim_zeros', '__return_true' );

            add_action( 'woocommerce_before_shop_loop', [ $this, 'update_shop_loop_result_count' ], 10 );
            add_action( 'woocommerce_before_shop_loop', [ $this, 'add_shop_filter_actives_bar' ], 150 );
            add_action( 'woocommerce_after_shop_loop', [ $this, 'update_woocommerce_pagination' ], 5 );

            add_filter( 'woocommerce_sale_flash', [ $this, 'add_percentage_to_sale_badge' ], 99, 3 );

            add_filter( 'loop_shop_per_page', [ $this, 'set_loop_shop_per_page' ], 20 );
            add_filter( 'woocommerce_product_query', [ $this, 'set_shop_per_page_query' ], 20 );
            
            // Change number of products(columns) per row(line)
            add_filter( 'loop_shop_columns', [ $this, 'tmcore_loop_columns' ] );

            // Search by sku
            add_filter( 'pre_get_posts', [ $this, 'tmcore_product_search_by_sku' ] );

            add_filter( 'woocommerce_product_get_rating_html', array( $this, 'product_get_rating_html' ), 10, 2);

            add_action( 'woocommerce_grouped_product_list_before_quantity', array( $this, 'grouped_product_thumbnail' ), 10 );

            remove_action( 'woocommerce_before_single_product_summary', 'woocommerce_show_product_sale_flash', 10 );
            add_action( 'woocommerce_single_product_summary', 'woocommerce_show_product_sale_flash', 1 );

            add_action( 'woocommerce_before_single_product_summary', [ $this, 'tmcore_product_summary_sidebar' ], 10 );

            add_action( 'woocommerce_single_product_summary', [ $this, 'wrap_price_rating_start' ], 9 );
            add_action( 'woocommerce_single_product_summary', [ $this, 'wrap_price_rating_end' ], 11 );
            
            add_action( 'woocommerce_single_product_summary', [ $this, 'live_view_visitors' ], 12 );
            add_action( 'woocommerce_single_product_summary', [ $this, 'product_sold_incart_text' ], 13 );

            add_filter( 'woocommerce_get_availability_text', [ $this, 'change_availability_text' ], 20, 2 );
            add_filter( 'wc_get_template', array( $this, 'tmcore_replace_stock_template' ), 30, 3 );

            if ( isset( $this->shopkit_settings['lazy_load_product_images'] ) && $this->shopkit_settings['lazy_load_product_images'] ) {
                remove_action( 'woocommerce_before_shop_loop_item_title', 'woocommerce_template_loop_product_thumbnail', 10 );
                add_action( 'woocommerce_before_shop_loop_item_title', [ $this, 'tmcore_woocommerce_get_product_thumbnail' ], 10);
            }

            // Buy now
            if ( isset( $this->shopkit_settings['bt_extra_buynow'] ) && $this->shopkit_settings['bt_extra_buynow'] ) {
                add_action( 'woocommerce_after_add_to_cart_button', [ $this, 'buy_now_button_single' ] );
                add_filter( 'woocommerce_add_to_cart_redirect', [ $this, 'redirect_to_checkout' ], 99 );
                add_action( 'wc_ajax_tmcore_add_to_cart', [ $this, 'add_to_cart_variable' ] );
            }

            if ( isset( $this->shopkit_settings['bt_extra_countdown_single_product'] ) && $this->shopkit_settings['bt_extra_countdown_single_product'] ) {
                add_action( 'woocommerce_single_product_summary', [ $this, 'tmcore_woocommerce_time_sale' ], 25 );   
            }
            
            if ( isset( $this->shopkit_settings['bt_extra_countdown_catalog'] ) && $this->shopkit_settings['bt_extra_countdown_catalog'] ) {
                add_action( 'woocommerce_before_shop_loop_item_title', [ $this, 'tmcore_woocommerce_time_sale' ], 12 );
            }

            // Add to cart on image hover
            if ( isset( $this->shopkit_settings['product_addtocart_on_image_hover'] ) && $this->shopkit_settings['product_addtocart_on_image_hover'] ) {
                //add_action( 'base_archive_loop_product_action_over_image', array($this, 'woocommerce_template_loop_add_to_cart_wrap_start'), 19 );
                add_action( 'base_archive_loop_product_action_over_image', 'woocommerce_template_loop_add_to_cart', 20 );
                //add_action( 'base_archive_loop_product_action_over_image', array($this, 'woocommerce_template_loop_add_to_cart_wrap_end'), 21 );
            }

            $wishlist_with_cart = ( isset( $this->shopkit_settings['product_wishlist_with_cart'] ) && !empty($this->shopkit_settings['product_wishlist_with_cart']) ) ? $this->shopkit_settings['product_wishlist_with_cart'] : '';
            $quickview_with_cart = ( isset( $this->shopkit_settings['product_quickview_with_cart'] ) && !empty($this->shopkit_settings['product_quickview_with_cart']) ) ? $this->shopkit_settings['product_quickview_with_cart'] : '';
            $compare_with_cart = ( isset( $this->shopkit_settings['product_compare_with_cart'] ) && !empty($this->shopkit_settings['product_compare_with_cart']) ) ? $this->shopkit_settings['product_compare_with_cart'] : '';

            if (function_exists('woosw_init') && get_option('woosw_button_position_archive') == "0" && !empty($wishlist_with_cart)) {
                add_action( 'woocommerce_after_shop_loop_item', [$this, 'wishlist_button'], 11 );
            }
            if (function_exists('woosq_init') && get_option('woosq_button_position') == "0" && !empty($quickview_with_cart) ) {
                add_action('woocommerce_after_shop_loop_item', [$this, 'quickview_button'], 12);
            }
            if (function_exists('woosc_init') ) {
                add_filter('woosc_button_position_archive','__return_false');
                if(!empty($compare_with_cart)) {
                    add_action('woocommerce_after_shop_loop_item', [$this, 'compare_button'], 13);
                }                
            }

            add_filter( 'woocommerce_post_class', array($this, 'tmcore_add_loop_item_class'), 10, 3 );
            add_filter( 'woocommerce_loop_add_to_cart_link',  array( $this, 'filter_add_to_cart_link_link' ), 50, 3);

            add_action( 'woocommerce_no_products_found', array($this, 'tmcore_show_product_on_not_found' ), 15 );

            
            

            if ( isset( $this->shopkit_settings['product_custom_thumb_width_height'] ) && !empty($this->shopkit_settings['product_custom_thumb_width_height']) ) {
                // Customized product image size on shop page
                add_filter( 'woocommerce_get_image_size_thumbnail', function( $size ) {

                    $product_image_width = ( isset( $this->shopkit_settings['product_thumb_image_width'] ) && !empty($this->shopkit_settings['product_thumb_image_width']) ) ? $this->shopkit_settings['product_thumb_image_width'] : 300;
                    $product_image_height = ( isset( $this->shopkit_settings['product_thumb_image_height'] ) && !empty($this->shopkit_settings['product_thumb_image_height']) ) ? $this->shopkit_settings['product_thumb_image_height'] : 300;
                    $product_image_crop = ( isset( $this->shopkit_settings['product_thumb_image_crop'] ) && !empty($this->shopkit_settings['product_thumb_image_crop']) ) ? $this->shopkit_settings['product_thumb_image_crop'] : 0;

                    return array(
                        'width' => (int)$product_image_width,
                        'height' => (int)$product_image_height,
                        'crop' => (int)$product_image_crop,
                    );
                });
            }
            
            // Change cross sells columns and limit.
            add_filter( 'woocommerce_cross_sells_columns', array( $this, 'cross_sell_columns' ), 30 );
            add_filter( 'woocommerce_cross_sells_total', array( $this, 'cross_sell_limit' ), 30 );

        }
        

        

        
        /**
         * Sets up theme defaults and registers support for various WooCommerce features.
         *
         * Note that this function is hooked into the after_setup_theme hook, which
         * runs before the init hook. The init hook is too late for some features, such
         * as indicating support for post thumbnails.
         *
         * @return void
         * @since 2.4.0
         */
        public function woocommerce_setup() {
            
            add_theme_support(
                'woocommerce', apply_filters(
                    'tmcore_woocommerce_args', array(
                        'product_grid' => array(
                            'default_columns' => 4,
                            'default_rows'    => 3,
                            'min_columns'     => 1,
                            'max_columns'     => 6,
                            'min_rows'        => 1,
                        ),
                    )
                )
            );

            /**
             * Add 'tmcore_woocommerce_setup' action.
             *
             * @since  1.0.0
             */
            do_action('tmcore_woocommerce_setup');
        }

        private function init_shortcodes() {
            foreach ($this->list_shortcodes as $shortcode) {
                add_filter('shortcode_atts_' . $shortcode, array($this, 'set_shortcode_attributes'), 10, 3);
                add_action('woocommerce_shortcode_before_' . $shortcode . '_loop', array(
                    $this,
                    'style_loop_start'
                ));
                add_action('woocommerce_shortcode_before_' . $shortcode . '_loop', array(
                    $this,
                    'shortcode_loop_start'
                ));
                add_action('woocommerce_shortcode_after_' . $shortcode . '_loop', array($this, 'style_loop_end'));
                add_action('woocommerce_shortcode_after_' . $shortcode . '_loop', array(
                    $this,
                    'shortcode_loop_end'
                ));
            }
        }

        public function shortcode_loop_end($atts = array()) {
            
            if (isset($atts['product_layout'])) {
                if ($atts['product_layout'] === 'list') {
                    if (!empty($atts['show_rating'])) {
                        remove_filter('tmcore_product_template_arg', 'tmcore_woocommerce_list_show_rating_arg', 10);
                    }
                    remove_filter('wc_get_template_part', 'tmcore_woocommerce_change_template_path', 10);
                } elseif ($atts['product_layout'] === 'carousel') {
                    echo '</div></div>';
                } elseif ($atts['product_layout'] === 'list-carousel') {
                    if (!empty($atts['show_rating'])) {
                        remove_filter('tmcore_product_template_arg', 'tmcore_woocommerce_list_show_rating_arg', 10);
                    }
                    remove_filter('wc_get_template_part', 'tmcore_woocommerce_change_template_path', 10);
                    echo '</div></div>';
                }
            }

            if (!empty($atts['image_size'])) {
                remove_filter('woocommerce_product_get_image', array($this, 'set_image_size_list'), 10);
            }
        }

        public function shortcode_loop_start($atts = array()) {
            if (isset($atts['product_layout'])) {
                if ($atts['product_layout'] === 'list') {
                    if (!empty($atts['show_rating'])) {
                        add_filter('tmcore_product_template_arg', 'tmcore_woocommerce_list_show_rating_arg', 10, 1);
                    }
                    add_filter('wc_get_template_part', 'tmcore_woocommerce_change_template_path', 10, 3);
                    if (!empty($atts['image_size'])) {
                        $this->list_size = $atts['image_size'];
                        add_filter('woocommerce_product_get_image', array($this, 'set_image_size_list'), 10, 2);
                    }
                } elseif ($atts['product_layout'] === 'carousel') {
                    echo '<div class="woocommerce-carousel splide" data-settings=\'' . $atts['carousel_settings'] . '\'>';
                    echo '<div class="splide__track">';

                } elseif ($atts['product_layout'] === 'list-carousel') {
                    if (!empty($atts['show_rating'])) {
                        add_filter('tmcore_product_template_arg', 'tmcore_woocommerce_list_show_rating_arg', 10, 1);
                    }
                    add_filter('wc_get_template_part', 'tmcore_woocommerce_change_template_path', 10, 3);
                    echo '<div class="woocommerce-carousel splide" data-settings=\'' . $atts['carousel_settings'] . '\'>';
                    echo '<div class="splide__track">';
                }
            }
        }

        public function style_loop_start($atts = array()) {

            if (isset($atts['product_layout'])) {

                if ($atts['product_layout'] === 'list' || $atts['product_layout'] === 'list-carousel') {

                    if (!empty($atts['show_category'])) {
                        add_action('woocommerce_widget_product_item_start', 'tmcore_woocommerce_get_product_category', 5);
                    }

                    if (!empty($atts['show_button'])) {
                        add_action('woocommerce_widget_product_item_end', 'woocommerce_template_loop_add_to_cart', 50);
                    }

                    if (!empty($atts['show_except'])) {
                        add_action('woocommerce_widget_product_item_end', 'tmcore_woocommerce_get_product_short_description', 55);
                    }
                    if (!empty($atts['show_countdown'])) {
                        if ( isset( $this->shopkit_settings['bt_extra_countdown'] ) && $this->shopkit_settings['bt_extra_countdown'] ) {
                            add_action('woocommerce_widget_product_item_end', [ $this, 'tmcore_woocommerce_time_sale' ], 60);
                        }
                    }
                } else {
                    if (!empty($atts['show_countdown'])) {
                        if ( isset( $this->shopkit_settings['bt_extra_countdown'] ) && $this->shopkit_settings['bt_extra_countdown'] ) {
                            add_action( 'woocommerce_before_shop_loop_item_title', [ $this, 'tmcore_woocommerce_time_sale' ], 12 );
                        }
                    }
                }
            }

        }


        public function style_loop_end($atts = array()) {

            if (isset($atts['product_layout'])) {

                if ($atts['product_layout'] === 'list' || $atts['product_layout'] === 'list-carousel') {

                    if (!empty($atts['show_category'])) {
                        remove_action('woocommerce_widget_product_item_start', 'tmcore_woocommerce_get_product_category', 5);
                    }

                    if (!empty($atts['show_button'])) {
                        remove_action('woocommerce_widget_product_item_end', 'woocommerce_template_loop_add_to_cart', 50);
                    }

                    if (!empty($atts['show_except'])) {
                        remove_action('woocommerce_widget_product_item_end', 'tmcore_woocommerce_get_product_short_description', 55);
                    }
                    if (!empty($atts['show_countdown'])) {
                        //if ( isset( $this->shopkit_settings['bt_extra_countdown'] ) && $this->shopkit_settings['bt_extra_countdown'] ) {
                            remove_action('woocommerce_widget_product_item_end', [ $this, 'tmcore_woocommerce_time_sale' ], 60);
                        //}
                    }
                    
                } else {
                    if (!empty($atts['show_countdown'])) {
                        //if ( isset( $this->shopkit_settings['bt_extra_countdown'] ) && $this->shopkit_settings['bt_extra_countdown'] ) {
                            remove_action( 'woocommerce_before_shop_loop_item_title', [ $this, 'tmcore_woocommerce_time_sale' ], 12 );
                        //}
                    }
                }
            }

        }

        public function set_shortcode_attributes($out, $pairs, $atts) {
            $out = wp_parse_args($atts, $out);

            return $out;
        }


        /**
         * Add WooCommerce specific classes to the body tag
         *
         * @param array $classes css classes applied to the body tag.
         *
         * @return array $classes modified to include 'woocommerce-active' class
         */
        public function woocommerce_body_class($classes) {
            $classes[] = 'woocommerce-active';

            $style   = '1';
            
            $classes[] = 'product-style-' . $style;
            if ($style == 1 || $style == 2) {
                $classes[] = 'products-no-gutter';
            }

            return $classes;
        }

        /**
         * WooCommerce specific scripts & stylesheets
         *
         * @since 1.0.0
         */
        public function action_enqueue_scripts() {

            wp_enqueue_style('tmcore-woocommerce-style',  TMCORE_URL . "assets/css/woocommerce/woocommerce{$this->suffix}.css", array(), TMCORE_VERSION );
            wp_style_add_data('tmcore-woocommerce-style', 'rtl', 'replace');

            //wp_enqueue_script('tmcore-input-quantity',  TMCORE_URL . "assets/js/woocommerce/quantity{$this->suffix}.js", array('jquery'), TMCORE_VERSION, true);

            wp_enqueue_script('tmcore-woocommerce-main',  TMCORE_URL . "assets/js/woocommerce/main{$this->suffix}.js", array('jquery'), TMCORE_VERSION, true);

            wp_register_script( 'tmcore-common-archive', TMCORE_URL . "assets/js/woocommerce/common-archive{$this->suffix}.js", [
                'jquery',
                'perfect-scrollbar',
            ], TMCORE_VERSION, true );
            wp_localize_script(
                'tmcore-common-archive',
                'tmCoreShop',
                array(
                    'total'            => wc_get_loop_prop( 'total' ),
                    'per_page'         => wc_get_loop_prop( 'per_page' ),
                    'current'          => wc_get_loop_prop( 'current_page' ),
                    'pagination_type'  => webapp()->option( 'infinite_pagination_type', 'load-more' ),
                    'is_collapsable'   => true,
                )
            );
            wp_enqueue_script( 'tmcore-common-archive' );
            
            $notifications_array = $this->product_purchased_notifications();
            if (!empty($notifications_array)) {
                wp_enqueue_script('tmc-notifications', TMCORE_URL . '/assets/js/notifications.min.js', array('jquery'), TMCORE_VERSION, true);
                wp_add_inline_script('tmc-notifications', 'var tmcore_notification_count=' . count($notifications_array) . '; var tmcore_notification=' . json_encode($notifications_array), 'before');
            }

            wp_enqueue_style( 'base-splide', TMCORE_URL . "assets/css/splide/splide.min.css" );
            wp_enqueue_script('base-splide', TMCORE_URL . "assets/js/splide/splide.min.js",  array(), TMCORE_VERSION, true); 
            wp_enqueue_script('base-splide-grid', TMCORE_URL . "assets/js/splide/splide-extension-grid.min.js",  array( 'base-splide' ), TMCORE_VERSION, true); 

        }


        /**
         * Register Sidebar for WooCommerce.
         *
         * @since  1.0.0
         */
        public function widgets_init() {
            register_sidebar(array(
                'name'          => __('WooCommerce Shop Sidebar', 'templatemela-core'),
                'id'            => 'sidebar-woocommerce-shop',
                'description'   => __('Add widgets here to appear in your sidebar product archives pages.', 'templatemela-core'),
                'before_widget' => '<div id="%1$s" class="widget %2$s">',
                'after_widget'  => '</div>',
                'before_title'  => '<span class="gamma widget-title">',
                'after_title'   => '</span>',
            ));
            register_sidebar(array(
                'name'          => __('WooCommerce Product Page Sidebar', 'templatemela-core'),
                'id'            => 'sidebar-woocommerce-product-page',
                'description'   => __('Add widgets here to appear in your sidebar on product details pages.', 'templatemela-core'),
                'before_widget' => '<div id="%1$s" class="widget %2$s">',
                'after_widget'  => '</div>',
                'before_title'  => '<span class="gamma widget-title">',
                'after_title'   => '</span>',
            ));
            register_sidebar(array(
                'name'          => __('WooCommerce Product Summary Sidebar', 'templatemela-core'),
                'id'            => 'sidebar-woocommerce-product-summary',
                'description'   => __('Add widgets here to appear in your sidebar on product details pages.', 'templatemela-core'),
                'before_widget' => '<div id="%1$s" class="widget %2$s">',
                'after_widget'  => '</div>',
                'before_title'  => '<span class="gamma widget-title">',
                'after_title'   => '</span>',
            ));
        }

        /**
         * Set Sidebar for WooCommerce.
         *
         * @since  1.0.0
         */
        public function set_sidebar($name) {
            $layout = 'default';
            if (is_shop() || is_product_taxonomy() || is_product_category() || is_product_tag()) {
                if (is_active_sidebar('sidebar-woocommerce-shop') && $layout == 'default') {
                    $name = 'sidebar-woocommerce-shop';
                } else {
                    $name = '';
                }
            }
            if (is_product()) {
                if (is_active_sidebar('sidebar-woocommerce-product-page')) {
                    $name = 'sidebar-woocommerce-product-page';
                } else {
                    $name = '';
                }
            }
            return $name;
        }

        /**
         * Active Filters Shop Bar.
         *
         * @since  1.0.0
         */
        public function add_shop_filter_actives_bar() {
            if ( webapp()->option( 'product_archive_shop_filter_active_top' ) ) {
                global $wp_query;
                if ( 0 === $wp_query->found_posts || ! woocommerce_products_will_display() ) {
                    return;
                }
            ?>
            <div id="active-filters-bar" class="active-filters-bar">
                <?php echo tmcore_get_remove_active_filter_links( $_GET ); ?>
            </div>
            <?php
            }
        }

        /**
         * shows percentage in flash sales
         */
        public function add_percentage_to_sale_badge( $html, $post, $product ) {

            if( $product->is_type('variable')){
                $percentages = array();
            
                // Get all variation prices
                $prices = $product->get_variation_prices();
            
                // Loop through variation prices
                foreach( $prices['price'] as $key => $price ){
                    // Only on sale variations
                    if( $prices['regular_price'][$key] !== $price ){
                        // Calculate and set in the array the percentage for each variation on sale
                        $percentages[] = round( 100 - ( floatval($prices['sale_price'][$key]) / floatval($prices['regular_price'][$key]) * 100 ) );
                    }
                }
                // We keep the highest value
                $percentage = max($percentages) . '%';
            
            } elseif( $product->is_type('grouped') ){
                $percentages = array();
            
                // Get all variation prices
                $children_ids = $product->get_children();
            
                // Loop through variation prices
                foreach( $children_ids as $child_id ){
                    $child_product = wc_get_product($child_id);

                    if( $child_product instanceof WC_Product ) {
                        
                        $regular_price = (float) $child_product->get_regular_price();
                        $sale_price    = (float) $child_product->get_sale_price();
                
                        if ( $sale_price != 0 || ! empty($sale_price) ) {
                            // Calculate and set in the array the percentage for each child on sale
                            $percentages[] = round(100 - ($sale_price / $regular_price * 100));
                        }
                    }
                }
                // We keep the highest value
                $percentage = max($percentages) . '%';
            
            } else {
                $regular_price = (float) $product->get_regular_price();
                $sale_price    = (float) $product->get_sale_price();
            
                if ( $sale_price != 0 || ! empty($sale_price) ) {
                    $percentage    = round(100 - ($sale_price / $regular_price * 100)) . '%';
                } else {
                    return $html;
                }
            }
            
            if ( isset( $this->shopkit_settings['product_sale_label_custom'] ) && !empty($this->shopkit_settings['product_sale_label_custom']) ) {


                $label = ( isset( $this->shopkit_settings['product_sale_label'] ) && !empty($this->shopkit_settings['product_sale_label']) ) ? $this->shopkit_settings['product_sale_label'] : '-{percentage}';               
                $label = str_replace('{percentage}', $percentage, $label);

                $flash_sale = '';
                $time_sale = get_post_meta($product->get_id(), '_sale_price_dates_to', true);
                if ($time_sale) { $flash_sale = 'flash-sale'; }
                //$off_text =  ' '. esc_html__( 'OFF', 'templatemela-core' );
                //return '<span class="onsale">' . ' - ' . $percentage .  $off_text . '</span>';
                return '<div class="product-onsale">
                            <span aria-hidden="true" class="onsale '. $flash_sale .'">' . $label . '</span>
                            <span class="screen-reader-text">'. esc_html__( 'Product on sale', 'templatemela-core' ) .'</span>
                        </div>';
            } else {
                return $html;
            }


            

        }
        
        /**
         * Change number of products that are displayed per page (shop page)
         */
        public function set_shop_per_page_query( $query ) {
            // Return the number of products you wanna show per page.
            if ( $query->is_main_query() && ( $query->get( 'wc_query' ) === 'product_query' ) ) {
                if ( isset( $this->shopkit_settings['bt_products_per_page'] ) && $this->shopkit_settings['bt_products_per_page'] ) {
                    $per_page = $this->shopkit_settings['bt_products_per_page'];
                }
                $per_page = !empty( $per_page ) ? $per_page : 12;
                $query->set( 'posts_per_page', $per_page );
            }
        }

        /**
         * Change number of products that are displayed per page (shop page - during ajax pagination )
         */
        public function set_loop_shop_per_page() {
            if ( isset( $this->shopkit_settings['bt_products_per_page'] ) && $this->shopkit_settings['bt_products_per_page'] ) {
                $per_page = $this->shopkit_settings['bt_products_per_page'];
            }
            $per_page = !empty( $per_page ) ? $per_page : 12;
            return isset( $_GET['product_per_page'] ) ? wc_clean( $_GET['product_per_page'] ) : $per_page;
        }

        /**
         * Change number of products(columns) per row(line)
         *
         * @access public
         */
        public function tmcore_loop_columns() {
            if ( isset( $this->shopkit_settings['bt_products_per_row'] ) && $this->shopkit_settings['bt_products_per_row'] ) {
                $per_page = $this->shopkit_settings['bt_products_per_row'];
            }
            $per_page = !empty( $per_page ) ? $per_page : 4;
            return $per_page;
        }

        /**
         * Update result count template function
         *
         * @since 1.0.0
         */
        public function update_shop_loop_result_count() {
            if ( is_shop() || is_product_taxonomy() || is_product_category() || is_product_tag() ) {
                add_filter( 'wc_get_template', array( $this, 'tmcore_shop_loop_result_count' ), 30, 2 );
            }
        }

        /**
         * Result count override
         *
         * @param  string $template template.
         * @param  string $template_name template name.
         * @return string
         */
        public function tmcore_shop_loop_result_count( $template, $template_name ) {
            $old_template = $template;
            if ( 'loop/result-count.php' === $template_name ) {
                $template = TMCORE_PATH . 'includes/templates/result-count.php';
            }
            return $template;
        }

        /**
         * Remove woocommerce_show_product_images and add custom gallery template function
         *
         * @since 1.0.0
         */
        public function update_woocommerce_pagination() {
            if ( is_shop() || is_product_taxonomy() || is_product_category() || is_product_tag() ) {
                add_filter( 'wc_get_template', array( $this, 'tmcore_shop_loop_pagination' ), 30, 2 );
            }
        }

        /**
         * Gallery template part override
         *
         * @param  string $template template.
         * @param  string $template_name template name.
         * @return string
         */
        public function tmcore_shop_loop_pagination( $template, $template_name ) {
            $old_template = $template;
            if ( 'loop/pagination.php' === $template_name ) {
                $template = TMCORE_PATH . 'includes/templates/pagination.php';
            }
            return $template;
        }

        /**
         * Display rating for all products
        **/
        public function product_get_rating_html($rating_html, $rating) {
            global $product;
            $rating_html  = '<div class="star-rating-block">';	
            $rating_html  .= '<div class="star-rating">';
            $rating_html .= '<span style="width:' . (($rating / 5) * 100) . '%"><strong class="rating">' . $rating . '</strong> ' . esc_attr__('out of 5', 'templatemela-core') . '</span>';
            $rating_html .= '</div>';
            $rating_html .= '</div>';
            return $rating_html;
        }

        /**
         * Add woocommerce grouped product thumbnail
         */
        public function grouped_product_thumbnail() {
            global $product;
            echo '<td class="woocommerce-grouped-product-list-item__thumbnail">';
            echo '<a href="' . get_the_permalink( $product->get_id() ) . '">' . wp_get_attachment_image( get_post_meta( $product->get_id(), '_thumbnail_id', true ), 'woocommerce_gallery_thumbnail' ) . '</a>';
            echo '</td>';
        }

        /**
         * Wrap Price and Rating
         */
        public function wrap_price_rating_start() {
            echo '<div class="wrap_price_rating">';
        }

        /**
         * Wrap Price and Rating
         */
        public function wrap_price_rating_end() {
            echo '</div>';
        }

        /**
         * Product summary sidebar
         */
        public function tmcore_product_summary_sidebar() {
            
            if ( is_product() && is_active_sidebar( 'sidebar-woocommerce-product-summary' ) ) {
                echo '<div class="summary-sidebar">';
                    dynamic_sidebar('sidebar-woocommerce-product-summary', array ());
                echo '</div>';
            }
            
        }

        public function tmcore_woocommerce_get_product_thumbnail() {

            if ( \Elementor\Plugin::$instance->editor->is_edit_mode() ) {
                return;
            } 
            
            global $post, $woocommerce;
            $output = '';
            $size = 'woocommerce_thumbnail';


            // Add a temporary filter to modify image attributes
            if ( !isset( $this->shopkit_settings['responsive_product_images_srcset'] ) || !$this->shopkit_settings['responsive_product_images_srcset'] ) {
                add_filter('wp_get_attachment_image_attributes', [ $this, 'updateImageAttributes' ], 10, 3);
            }

            if ( has_post_thumbnail() ) {
                $src = get_the_post_thumbnail_url( $post->ID, $size );
                $output .= get_the_post_thumbnail( $post->ID, $size, [
                    'data-src' => get_the_post_thumbnail_url( $post->ID,$size ),
                    'src' => TMCORE_URL . 'assets/images/loading.gif',
                    //'data-srcset' => wp_get_attachment_image_srcset( get_post_thumbnail_id($post->ID) ),
                    'class' => 'tmcore-lazy-load',
                ]);
            } else {
                $output .= wc_placeholder_img( $size );
            }

            // Remove the filter to avoid affecting other images
            if ( !isset( $this->shopkit_settings['responsive_product_images_srcset'] ) || !$this->shopkit_settings['responsive_product_images_srcset'] ) {
                remove_filter('wp_get_attachment_image_attributes', [ $this, 'updateImageAttributes' ], 10);
            }

            echo $output;

        }
        public function updateImageAttributes( $attr, $attachment, $size ){
            unset($attr['srcset']); // Remove the srcset attribute
            // Check if we are in the cart
            //if (is_cart()) {}
            return $attr;
        }
        
        /**
         * Search by sku
         */
        public function tmcore_product_search_by_sku( $query ) {

            // Check if on frontend and is main query
            if (!is_admin() && $query->is_main_query()) {
                if ($query->is_search()) {
                    // Get the search term
                    $search_term = $query->get("s");
        
                    // Add SKU to search query
                    add_filter("posts_join", [ $this, "tmcore_sku_search_join"], 10, 2);
                    add_filter("posts_where", [ $this, "tmcore_sku_search_where"], 10, 2);
        
                    // Include only products in search
                    if ( isset($_REQUEST['post_type']) && $_REQUEST['post_type'] == 'product') {
                        $query->set("post_type", "product");
                    }
                    // Make sure results are distinct
                    $query->set("distinct", "DISTINCT");
                }
            } 
        }
        /**
         * Search by Sku
         */
        public function tmcore_sku_search_join($join, $query) {
            global $wpdb;
 
            // Check if it's a WooCommerce product search
            if ($query->is_search()) {
                $join .= " LEFT JOIN {$wpdb->postmeta} as pm ON {$wpdb->posts}.ID = pm.post_id AND pm.meta_key = '_sku' ";
            }
            return $join;
        }
        /**
         * Search by Sku
         */
        public function tmcore_sku_search_where($where, $query) {
            global $wpdb;
 
            // Check if it's a WooCommerce product search
            if ($query->is_search()) {
                $search_term = $query->get("s");
        
                // Searching for SKUs
                $where .=
                    " OR (pm.meta_key = '_sku' AND pm.meta_value LIKE '%" .
                    esc_sql($search_term) .
                    "%')";
            }
            return $where;
        }
        

        /**
         * Add sale time to the product
         */
        public function tmcore_woocommerce_time_sale() {
            /**
             * @var $product WC_Product
             */
            global $product;
            $called_action_hook = current_filter();
            if (!$product->is_on_sale()) {
                return;
            }


            $time_sale = get_post_meta($product->get_id(), '_sale_price_dates_to', true);
            if ($time_sale) {
                $time_sale += (get_option('gmt_offset') * HOUR_IN_SECONDS);
                
                wp_enqueue_script('tmcore-countdown');

                $countdown_style = isset($this->shopkit_settings['countdown_style_global'])  ?  $this->shopkit_settings['countdown_style_global'] : 'style_1';
                
                if ( is_product() && $called_action_hook == 'woocommerce_single_product_summary' ) {
                    $countdown_style = isset($this->shopkit_settings['countdown_style_single_product'])  ? 'single_' .$this->shopkit_settings['countdown_style_single_product'] : 'style_1';
                    $countdown_label = isset($this->shopkit_settings['countdown_label'])  ? $this->shopkit_settings['countdown_label'] : 'Hurry up ! Sale end in:';
                }
                ?>
                <div class="time-sale <?php echo esc_attr( $countdown_style );?>">
                     <?php if ( is_product() && $called_action_hook == 'woocommerce_single_product_summary' ) { ?>
                    <div class="deal-text">
                        <span class="countdown-timer-text"><?php echo esc_html( $countdown_label ); ?></span>
                    </div>
                    <?php } ?>
                    <div class="tmcore-countdown" data-countdown="true" data-date="<?php echo esc_html($time_sale); ?>">
                        <div class="countdown-item">
                            <span class="countdown-digits countdown-days"></span>
                            <span class="countdown-label"><?php echo esc_html__('DAYS', 'templatemela-core') ?></span>
                        </div>
                        <div class="countdown-item">
                            <span class="countdown-digits countdown-hours"></span>
                            <span class="countdown-label"><?php echo esc_html__('HRS', 'templatemela-core') ?></span>
                        </div>
                        <div class="countdown-item">
                            <span class="countdown-digits countdown-minutes"></span>
                            <span class="countdown-label"><?php echo esc_html__('MIN', 'templatemela-core') ?></span>
                        </div>
                        <div class="countdown-item">
                            <span class="countdown-digits countdown-seconds"></span>
                            <span class="countdown-label"><?php echo esc_html__('SEC', 'templatemela-core') ?></span>
                        </div>
                    </div>
                </div>
                <?php
            }
        }


        /**
         * Add woocommerce sold products count
         */
        public function product_sold_incart_text() {

            global $product;

            if ( (!isset($this->shopkit_settings['bt_product_sold']) || '1' != $this->shopkit_settings['bt_product_sold']) && (!isset($this->shopkit_settings['bt_product_incart']) || '1' != $this->shopkit_settings['bt_product_incart'])  ) {
                return;
                        
            }
            if( !$product || "outofstock" === $product->get_stock_status() ) {
                return;
            }
            
            $product_type = $product->get_type();
            $types_allow = apply_filters('tmcore_types_allow_product_sold', array('simple', 'variable', 'variation'));

            if (empty($types_allow) || in_array($product_type, $types_allow)) {
                $product_id = $product_type == 'variation' ? $product->get_parent_id() : $product->get_id();
        
                $bt_product_sold = isset($this->shopkit_settings['bt_product_sold']) && $this->shopkit_settings['bt_product_sold'] == '1' ? $this->shopkit_settings['bt_product_sold'] : null;
                $bt_product_incart = isset($this->shopkit_settings['bt_product_incart']) && $this->shopkit_settings['bt_product_incart'] == '1'  ? $this->shopkit_settings['bt_product_incart'] : null;
                $bt_product_combined = isset($this->shopkit_settings['bt_product_sold_incart_combined']) && $this->shopkit_settings['bt_product_sold_incart_combined'] == '1'  ? $this->shopkit_settings['bt_product_sold_incart_combined'] : null;                    
                
                if ( '1' == $bt_product_sold && '1' == $bt_product_incart && '1' == $bt_product_combined ) {
                    echo '<div class="check-incart-sold-wrap"><div class="check-incart-sold"><div class="tmcore-scroll">';
                }

                if ( isset($this->shopkit_settings['bt_product_sold']) && '1' == $this->shopkit_settings['bt_product_sold'] ) {

                    $key_name = 'tmcore_product_sold_' . $product_id;
                    $product_sold = get_transient($key_name);

                    if (!$product_sold) {
                        $min = isset($this->shopkit_settings['bt_product_sold_min']) && (int) $this->shopkit_settings['bt_product_sold_min'] ? (int) $this->shopkit_settings['bt_product_sold_min'] : 1;
                        $max = isset($this->shopkit_settings['bt_product_sold_max']) && (int) $this->shopkit_settings['bt_product_sold_max'] ? (int) $this->shopkit_settings['bt_product_sold_max'] : 20;
                        $sold = rand($min, $max);
                        if($sold == 1 ) { $sold = 2; };

                        $min_time = isset($this->shopkit_settings['bt_product_sold_time_min']) && (int) $this->shopkit_settings['bt_product_sold_time_min'] ? (int) $this->shopkit_settings['bt_product_sold_time_min'] : 1;
                        $max_time = isset($this->shopkit_settings['bt_product_sold_time_max']) && (int) $this->shopkit_settings['bt_product_sold_time_max'] ? (int) $this->shopkit_settings['bt_product_sold_time_max'] : 20;
                        $times = rand($min_time, $max_time);

                        $live_time = isset($this->shopkit_settings['bt_product_sold_live_time']) && (int) $this->shopkit_settings['bt_product_sold_live_time'] ? (int) $this->shopkit_settings['bt_product_sold_live_time'] : 36000;

                        $product_sold_data = '<div class="tmcore-sold-incart tmcore-last-sold">';
                        $product_sold_data .= '<i class="icon-fire"><svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="18" fill="#da3f3f" height="18" viewBox="0 0 512 512"><g id="tmcore-ignore"></g><path d="M160.504 512c-34.123-71.004-15.951-111.689 10.275-150.022 28.72-41.978 36.122-83.533 36.122-83.533s22.577 29.35 13.546 75.254c39.886-44.4 47.412-115.14 41.391-142.232 90.157 63.006 128.688 199.428 76.761 300.533 276.186-156.266 68.699-390.086 32.577-416.425 12.041 26.338 14.324 70.925-10 92.563-41.176-156.138-142.986-188.138-142.986-188.138 12.041 80.522-43.648 168.572-97.348 234.365-1.887-32.108-3.891-54.264-20.775-84.99-3.79 58.328-48.366 105.874-60.437 164.314-16.351 79.143 12.248 137.090 120.874 198.311z"></path></svg></i>';

                        $product_sold_data .= sprintf(
                            _n( '%s products sold in last %s hour', '%s products sold in last %s hours', $times, 'templatemela-core' ),
                            $sold,
                            $times
                        );
                            
                        $product_sold_data .='</div>';

                        $product_sold = apply_filters('tmcore_product_sold_content', $product_sold_data, $product_id);

                        set_transient($key_name, $product_sold, $live_time);   
                    }
                    echo $product_sold ? $product_sold : '';
                }

                if ( isset($this->shopkit_settings['bt_product_incart']) && '1' == $this->shopkit_settings['bt_product_incart'] ) {

                $key_name = 'tmcore_product_incart_' . $product_id;
                $product_incart = get_transient($key_name);
                
                if (!$product_incart) {

                    $min = isset($this->shopkit_settings['bt_product_incart_min']) && (int) $this->shopkit_settings['bt_product_incart_min'] ? (int) $this->shopkit_settings['bt_product_incart_min'] : 1;
                    $max = isset($this->shopkit_settings['bt_product_incart_max']) && (int) $this->shopkit_settings['bt_product_incart_max'] ? (int) $this->shopkit_settings['bt_product_incart_max'] : 20;
                    $in_cart = rand($min, $max);

                    $live_time = isset($this->shopkit_settings['bt_product_incart_live_time']) && (int) $this->shopkit_settings['bt_product_incart_live_time'] ? (int) $this->shopkit_settings['bt_product_incart_live_time'] : 36000;

                    $product_incart_data = '<div class="tmcore-sold-incart tmcore-in-cart">';
                    $product_incart_data .= '<i class="icon-fire"><svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="18" fill="#da3f3f" height="18" viewBox="0 0 512 512"><g id="tmcore-ignore"></g><path d="M160.504 512c-34.123-71.004-15.951-111.689 10.275-150.022 28.72-41.978 36.122-83.533 36.122-83.533s22.577 29.35 13.546 75.254c39.886-44.4 47.412-115.14 41.391-142.232 90.157 63.006 128.688 199.428 76.761 300.533 276.186-156.266 68.699-390.086 32.577-416.425 12.041 26.338 14.324 70.925-10 92.563-41.176-156.138-142.986-188.138-142.986-188.138 12.041 80.522-43.648 168.572-97.348 234.365-1.887-32.108-3.891-54.264-20.775-84.99-3.79 58.328-48.366 105.874-60.437 164.314-16.351 79.143 12.248 137.090 120.874 198.311z"></path></svg></i>';

                    $product_incart_data .= sprintf(
                        _n( 'Selling fast! %s person has this in their cart', 'Selling fast! Over %s people have this in their carts', $in_cart, 'templatemela-core' ),
                        $in_cart
                    );
                    $product_incart_data .='</div>';

                    $product_incart = apply_filters('tmcore_product_incart_content', $product_incart_data, $product_id);

                    set_transient($key_name, $product_incart, $live_time);

                    }
                    echo $product_incart ? $product_incart : '';
                }
                
                if ( '1' == $bt_product_sold && '1' == $bt_product_incart && '1' == $bt_product_combined ) {
                    echo '</div></div></div>';
                }
                
            }
        }
        
        /**
         * Add woocommerce live viewing visitors
         */
        public function live_view_visitors() {

            if ( empty( $this->shopkit_settings['bt_product_live_view'] ) ) {
                return;
            }

            $range_min = isset($this->shopkit_settings['bt_product_live_view_min'])  ? $this->shopkit_settings['bt_product_live_view_min'] : 20;
            $range_max = isset($this->shopkit_settings['bt_product_live_view_max'])  ? $this->shopkit_settings['bt_product_live_view_max'] : 30;
            $range = array( $range_min, $range_max );
    
            if ( ! empty( $range ) ) {
                $min = $range[0];
                $max = $range[1];
    
                $total_visitors = mt_rand( $min, $max );

                $settings= [
                    'min'      => $min,
                    'max'      => $max,
                    'duration' => 10000, // in ms.
                    'labels'   => [
                        'singular' => esc_html( _n( '%s person is viewing this right now', '%s people are viewing this right now', 1, 'templatemela-core' ) ),
                        'plural'   => esc_html( _n( '%s person is viewing this right now', '%s people are viewing this right now', 99, 'templatemela-core' ) ),
                    ],
                ];

                ?>
                <div id="tmcore-live-viewing-visitors" class="tmcore-live-viewing-visitors" data-settings="<?php echo esc_attr( json_encode( $settings ) ); ?>">
                    <i class="tmcore-icon-eye eye"></i>    
                    <div class="text-wrapper">
                    <?php
                        echo sprintf(
                            esc_html( _n( '%s person is viewing this right now', '%s people are viewing this right now', $total_visitors, 'templatemela-core' ) ),
                            '<span class="count">' . $total_visitors . '</span>'
                        );
                    ?>
                    </div>
                    
                </div>
                <?php
            }
        }

        /**
         * @param string      $availability
         * @param \WC_Product $product
         *
         * @return string
         */
        public function change_availability_text( $availability, $product ) {
            if ( ! $product->is_in_stock() ) {
                $availability = __( 'Out of stock', 'templatemela-core' );
            } elseif ( $product->managing_stock() && $product->is_on_backorder( 1 ) ) {
                $availability = $product->backorders_require_notification() ? __( 'Available on backorder', 'templatemela-core' ) : '';
            } elseif ( ! $product->managing_stock() && $product->is_on_backorder( 1 ) ) {
                $availability = __( 'Available on backorder', 'templatemela-core' );
            } elseif ( $product->managing_stock() ) {
                $availability        = __( 'In stock', 'woocommerce' );
                $stock_amount        = $product->get_stock_quantity();
                $low_stock_amount    = wc_get_low_stock_amount( $product );
                $stock_quantity_html = '<span class="value">' . wc_format_stock_quantity_for_display( $stock_amount, $product ) . '</span>';
                $stock_quantity      = wc_format_stock_quantity_for_display( $stock_amount, $product );

                switch ( get_option( 'woocommerce_stock_format' ) ) {
                    case 'low_amount':
                        if ( $stock_amount <= $low_stock_amount ) {
                            $items_text = '<span class="value">';
                            /* translators: %s: stock amount */
                            $items_text .= sprintf(
                                _n( '%s item', '%s items', $stock_amount, 'templatemela-core' ),
                                $stock_quantity
                            );
                            $items_text .= '</span>';
                            /* translators: %s: stock amount + items text */
                            $availability = sprintf(
                                _n( 'Hurry! Only %s is left in stock', 'Hurry! Only %s are left in stock', $stock_amount, 'templatemela-core' ),
                                $items_text
                            );
                        }
                        break;
                    case '':
                        if ( $stock_amount <= $low_stock_amount ) {
                            $items_text = '<span class="value">';
                            /* translators: %s: stock amount */
                            $items_text .= sprintf(
                                _n( '%s item', '%s items', $stock_amount, 'templatemela-core' ),
                                $stock_quantity
                            );
                            $items_text .= '</span>';
                            /* translators: %s: stock amount + items text */
                            $availability = sprintf(
                                _n( 'Hurry! Only %s is left in stock', 'Hurry! Only %s are left in stock', $stock_amount, 'templatemela-core' ),
                                $items_text
                            );
                        } else {
                            /* translators: %s: stock amount */
                            $availability = sprintf( __( '%s in stock', 'woocommerce' ), $stock_quantity_html );
                        }
                        break;
                }

                if ( $product->backorders_allowed() && $product->backorders_require_notification() ) {
                    $availability .= ' ' . __( '(can be backordered)', 'woocommerce' );
                }
            } else {
                $availability = '';
            }

            return $availability;
        }

        /**
         * Stock count override
         *
         * @param  string $template template.
         * @param  string $template_name template name.
         * @return string
         */
        public function tmcore_replace_stock_template( $template, $template_name, $args ) {
            $old_template = $template;
            if ( 'single-product/stock.php' === $template_name ) {
                $template = TMCORE_PATH . 'includes/templates/stock.php';
            }
            return $template;
        }

        /**
         * Add buy now button on single product page
         */
        public function buy_now_button_single() {
            global $product;

            // conditional
            if ( $product->is_type( 'simple' ) || $product->is_type( 'variable' ) ) { ?>
				<button type="submit" name="tmcore-buy-now" value="<?php echo esc_attr( $product->get_id() ); ?>"
				        class="single_add_to_cart_button ajax_add_to_cart button alt button-buy-now <?php echo esc_attr( wc_wp_theme_get_element_class_name( 'button' ) ); ?>"
				        data-redirect="<?php echo esc_url( wc_get_checkout_url() ); ?>">
					<span><?php echo esc_html( $this->get_button_text() ); ?></span></button>
			    <?php
            }
        }
        
        public function get_button_text() {
            $button_text = isset($this->shopkit_settings['bt_extra_buynow_text_single'])  ? $this->shopkit_settings['bt_extra_buynow_text_single'] : null;

            if ( empty( $button_text ) ) {
                $button_text = __( 'Buy now', 'templatemela-core' );
            }
            return $button_text;
        }

        /**
         * Try redirect to checkout page after product added to cart.
         * This function fallback when ajax add to cart disabled.
         *
         * @param $url
         *
         * @return string
         */
        public function redirect_to_checkout( $url ) {
            if ( isset( $_REQUEST['tmcore-buy-now'] ) ) {
                return wc_get_checkout_url();
            }

            return $url;
        }

        /**
         * @see \WC_AJAX::add_to_cart()
         * @see \WC_Form_Handler::add_to_cart_action()
         */
        public function add_to_cart_variable() {
            // phpcs:disable WordPress.Security.NonceVerification.Missing
            if ( ! isset( $_POST['product_id'] ) ) {
                return;
            }

            $product_id        = apply_filters( 'woocommerce_add_to_cart_product_id', absint( $_POST['product_id'] ) );
            $product           = wc_get_product( $product_id );
            $quantity          = empty( $_POST['quantity'] ) ? 1 : wc_stock_amount( wp_unslash( $_POST['quantity'] ) );
            $passed_validation = apply_filters( 'woocommerce_add_to_cart_validation', true, $product_id, $quantity );
            $product_status    = get_post_status( $product_id );
            $variation_id      = ! empty( $_POST['variation_id'] ) ? absint( $_POST['variation_id'] ) : 0;
            $variation         = array();

            if ( $product && 'variation' === $product->get_type() && empty( $variation_id ) ) {
                $variation_id = $product_id;
                $product_id   = $product->get_parent_id();
            }

            foreach ( $_POST as $key => $value ) {
                if ( 'attribute_' !== substr( $key, 0, 10 ) ) {
                    continue;
                }

                $variation[ sanitize_title( wp_unslash( $key ) ) ] = wp_unslash( $value );
            }

            if ( $passed_validation && false !== \WC()->cart->add_to_cart( $product_id, $quantity, $variation_id, $variation ) && 'publish' === $product_status ) {

                do_action( 'woocommerce_ajax_added_to_cart', $product_id );

                if ( 'yes' === get_option( 'woocommerce_cart_redirect_after_add' ) ) {
                    wc_add_to_cart_message( array( $product_id => $quantity ), true );
                }

                \WC_AJAX::get_refreshed_fragments();

            } else {
                // If there was an error adding to the cart, redirect to the product page to show any errors.
                $data = array(
                    'error'       => true,
                    'product_url' => apply_filters( 'woocommerce_cart_redirect_after_error', get_permalink( $product_id ), $product_id ),
                );

                wp_send_json( $data );
            }
        }

        /**
         * Product purchased notifications
         *
         * @since 1.0.0
         */
        public function product_purchased_notifications() {
            
            if ( empty( $this->shopkit_settings['notification_popup'] ) ) {
                return;
            }

            $notification_data = array();
            for($i = 1; $i<=10; $i++) { // upto 10 notifications input from admin panel

                $product_id = isset($this->shopkit_settings['notification_popup_product_'.strval($i)]) ?  $this->shopkit_settings['notification_popup_product_'.strval($i)] : '';
                $product_info = isset($this->shopkit_settings['notification_popup_info_'.strval($i)]) ?  $this->shopkit_settings['notification_popup_info_'.strval($i)] : '';
                $product_time = isset($this->shopkit_settings['notification_popup_time_'.strval($i)]) ?  $this->shopkit_settings['notification_popup_time_'.strval($i)] : '';

                if(!empty($product_id)) {
                    $notification_data[] = array(
                        'product_info' => $product_info,
                        'product_time' => $product_time,
                        'id' => $product_id
                    );    
                }
            }
            $notification_data = array_filter($notification_data);
            if (!empty($notification_data)) {
                $return_array = array();
                foreach ($notification_data as $prodata) {
                    $prodata = (array) $prodata;
                    $product_obj = isset($prodata['id']) ? wc_get_product($prodata['id']) : null;
                    
                    if (!empty($product_obj)) {

                        $image_thumbnail_id = $product_obj ? $product_obj->get_image_id() : 0;
                        $image_size = apply_filters('woocommerce_gallery_image_size', 'woocommerce_gallery_thumbnail');

                        $product_id = $product_obj->get_id();
                        $product_title = $product_obj ? $product_obj->get_title() : '';
                        $product_url = $product_obj ? get_permalink( $product_id ) : '';
                        
                        $image = $image_thumbnail_id ? wp_get_attachment_image(
                            $image_thumbnail_id,
                            $image_size,
                        ) : wc_placeholder_img($image_size);
                        $return_array[] = array(
                            'product_image' => $image, 
                            'product_info' => $prodata['product_info'], 
                            'product_url' => $product_url,
                            'product_name' => $product_title,
                            'time_purchase' => $prodata['product_time'],
                            'product_data' => isset($prodata['id']) ? $prodata['id'] : 0
                        );
                    }
                }
                return $return_array;
            }
        }

        /**
         * Adds Arrow to add to cart button.
         *
         * @param string $button Current classes.
         * @param object $product Product object.
         * @param array  $args The Product args.
         */
        public function filter_add_to_cart_link_link( $button, $product, $args = array() ) {

            $show_button = true;
            if ( isset( $this->shopkit_settings['product_add_to_cart'] ) && $this->shopkit_settings['product_add_to_cart'] ) {
                $show_button = true;
            } else {
                if ( is_product_category() || is_shop() ) {
                    $show_button = false;
                } else {
                    $show_button = true;
                }
            }

            //if( $show_button == true ){

                $is_btn = webapp()->option( 'product_archive_button_style' );
                if ( isset( $is_btn ) && 'button' == $is_btn){
                    $product_btn_style = 'default';
                    if ( isset( $this->shopkit_settings['product_action_button_style'] ) && $this->shopkit_settings['product_action_button_style'] ) {
                        $product_btn_style = $this->shopkit_settings['product_action_button_style'];
                    }
                } else {
                    $product_btn_style = 'text';
                }
                
                
                $icon = '';
                $icon .= '<span class="add-to-cart-icon">';
                if ( $product->is_type( 'variable' ) || $product->is_type( 'grouped' ) ) {
                    $icon .= '<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" width="24px" height="24px" viewBox="0 0 24 24" fill="currentColor" version="1.1"><g id="surface1"><path d="M 14.527344 2.125 C 14.414062 2.183594 14.257812 2.324219 14.179688 2.433594 C 14.054688 2.617188 14.039062 2.667969 14.039062 2.996094 C 14.039062 3.3125 14.054688 3.378906 14.164062 3.542969 C 14.234375 3.640625 14.371094 3.777344 14.46875 3.84375 L 14.644531 3.960938 L 16.613281 3.976562 L 18.582031 3.988281 L 14.382812 8.195312 C 9.941406 12.632812 9.984375 12.585938 9.984375 13.007812 C 9.984375 13.203125 10.121094 13.550781 10.257812 13.707031 C 10.378906 13.847656 10.785156 14.015625 10.992188 14.015625 C 11.417969 14.011719 11.371094 14.054688 15.804688 9.617188 L 20.011719 5.417969 L 20.023438 7.386719 L 20.039062 9.355469 L 20.15625 9.53125 C 20.222656 9.628906 20.359375 9.765625 20.457031 9.835938 C 20.621094 9.945312 20.6875 9.960938 21 9.960938 C 21.3125 9.960938 21.378906 9.945312 21.542969 9.835938 C 21.640625 9.765625 21.777344 9.628906 21.84375 9.53125 L 21.960938 9.355469 L 21.960938 2.640625 L 21.835938 2.457031 C 21.765625 2.359375 21.628906 2.222656 21.53125 2.15625 L 21.355469 2.039062 L 18.046875 2.023438 L 14.738281 2.015625 Z M 14.527344 2.125 "/><path d="M 4.777344 4.035156 C 3.488281 4.226562 2.433594 5.183594 2.105469 6.460938 C 2.019531 6.796875 2.015625 7.035156 2.015625 13.003906 C 2.015625 19.914062 1.996094 19.460938 2.351562 20.203125 C 2.507812 20.53125 2.621094 20.675781 2.972656 21.027344 C 3.324219 21.378906 3.46875 21.492188 3.796875 21.648438 C 4.539062 22.003906 4.085938 21.984375 10.996094 21.984375 C 16.960938 21.984375 17.203125 21.980469 17.539062 21.894531 C 18.6875 21.601562 19.550781 20.75 19.902344 19.566406 C 19.976562 19.3125 19.988281 18.945312 20.007812 15.515625 C 20.015625 13.4375 20.007812 11.628906 19.992188 11.5 C 19.929688 11.011719 19.566406 10.648438 19.078125 10.601562 C 18.683594 10.566406 18.355469 10.738281 18.132812 11.101562 L 18.023438 11.273438 L 18 15.1875 C 17.976562 19.09375 17.976562 19.101562 17.875 19.292969 C 17.738281 19.546875 17.503906 19.773438 17.25 19.894531 L 17.039062 19.992188 L 4.945312 19.992188 L 4.726562 19.882812 C 4.472656 19.757812 4.199219 19.476562 4.085938 19.222656 C 4.011719 19.070312 4.007812 18.53125 4.007812 13.007812 L 4.007812 6.960938 L 4.105469 6.75 C 4.226562 6.496094 4.453125 6.261719 4.707031 6.125 C 4.898438 6.023438 4.921875 6.023438 8.347656 6 L 11.792969 5.976562 L 11.992188 5.84375 C 12.273438 5.664062 12.425781 5.40625 12.453125 5.078125 C 12.488281 4.703125 12.308594 4.355469 11.976562 4.148438 L 11.75 4.007812 L 8.394531 4.003906 C 6.546875 4 4.921875 4.011719 4.777344 4.035156 Z M 4.777344 4.035156 "/></g></svg>';
                } else {
                    $icon .= '<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" width="22px" height="22px" viewBox="0 0 22 22" fill="currentColor" version="1.1"><path d="M19.45 15.24a.86.86 0 00.848-.719l1.69-10.14a.86.86 0 00-.848-1H4.91L4.229.65A.86.86 0 003.395 0H.858a.86.86 0 100 1.719h1.865l.673 2.696L5.07 14.45v2.6a2.553 2.553 0 00-1.69 2.4A2.552 2.552 0 005.93 22c1.744 0 2.981-1.726 2.41-3.38h7.01c-.572 1.655.667 3.38 2.41 3.38a2.552 2.552 0 002.55-2.55 2.552 2.552 0 00-2.55-2.55H6.79v-1.66zm.676-10.141l-1.404 8.422H6.658L5.254 5.099zM6.76 19.45a.832.832 0 01-1.661 0 .832.832 0 011.661 0m11 .831a.832.832 0 010-1.661.832.832 0 010 1.661"></path></svg>';
                }
                $icon .= '</span>';    
            
                if ( 'is_btn' === $product_btn_style ) {
                    $button = sprintf(
                        '<a href="%s" data-quantity="%s" class="%s" %s>%s</a>',
                        esc_url( $product->add_to_cart_url() ),
                        esc_attr( isset( $args['quantity'] ) ? $args['quantity'] : 1 ),
                        esc_attr( isset( $args['class'] ) ? $args['class'] . ' ' . $product_btn_style  : 'button' . ' ' . $product_btn_style ),
                        isset( $args['attributes'] ) ? wc_implode_html_attributes( $args['attributes'] ) : '',
                        $icon .'<span class="cart-text">'. esc_html( $product->add_to_cart_text() ) . '</span>' . webapp()->get_icon( 'spinner' ) . '' . webapp()->get_icon( 'check' )
                    );
                } else {
                    $button = sprintf(
                        '<a href="%s" data-quantity="%s" class="%s" %s>%s</a>',
                        esc_url( $product->add_to_cart_url() ),
                        esc_attr( isset( $args['quantity'] ) ? $args['quantity'] : 1 ),
                        esc_attr( isset( $args['class'] ) ? $args['class'] . ' ' . $product_btn_style  : $product_btn_style ),
                        isset( $args['attributes'] ) ? wc_implode_html_attributes( $args['attributes'] ) : '',
                        $icon .'<span class="cart-text">'. esc_html( $product->add_to_cart_text() ) . '</span>' . webapp()->get_icon( 'arrow-right-alt' ) . '' . webapp()->get_icon( 'spinner' ) . '' . webapp()->get_icon( 'check' )
                    );
                }
                return $button;
            /* } else {
                return '';
            } */

            
        }

        public function tmcore_add_loop_item_class( $classes ) {
            
            //if ( 'product' == get_post_type() ) {

                $new_classes = '';

                $show_add_to_cart = ( isset( $this->shopkit_settings['product_add_to_cart'] ) && !empty($this->shopkit_settings['product_add_to_cart']) ) ? $this->shopkit_settings['product_add_to_cart'] : '';
                $wishlist_with_cart = ( isset( $this->shopkit_settings['product_wishlist_with_cart'] ) && !empty($this->shopkit_settings['product_wishlist_with_cart']) ) ? $this->shopkit_settings['product_wishlist_with_cart'] : '';
                $quickview_with_cart = ( isset( $this->shopkit_settings['product_quickview_with_cart'] ) && !empty($this->shopkit_settings['product_quickview_with_cart']) ) ? $this->shopkit_settings['product_quickview_with_cart'] : '';
                $compare_with_cart = ( isset( $this->shopkit_settings['product_compare_with_cart'] ) && !empty($this->shopkit_settings['product_compare_with_cart']) ) ? $this->shopkit_settings['product_compare_with_cart'] : '';
                
                
                if ( empty($show_add_to_cart) && empty($wishlist_with_cart) && empty($quickview_with_cart) && empty($compare_with_cart) ) {
                    $new_classes .= ' no-buttons';
                }
                if ( isset( $this->shopkit_settings['product_add_to_cart'] ) && $this->shopkit_settings['product_add_to_cart'] ) {
                    $new_classes .= ' cart-button-' . $this->shopkit_settings['product_add_to_cart'];
                } else {
                    $new_classes .= ' cart-button-hide';
                }
                if ( isset( $this->shopkit_settings['product_action_button_style'] ) && $this->shopkit_settings['product_action_button_style'] ) {
                    $new_classes .= ' action-style-' . $this->shopkit_settings['product_action_button_style'];
                }
                if ( isset( $this->shopkit_settings['product_hover_style'] ) && $this->shopkit_settings['product_hover_style'] ) {
                    $new_classes .= ' product-hover-' . $this->shopkit_settings['product_hover_style'] ;
                } else {
                    $new_classes .= ' product-hover-style1';
                }
                if ( isset( $this->shopkit_settings['product_image_hover_style'] ) && $this->shopkit_settings['product_image_hover_style'] ) {
                    $new_classes .= ' image-hover-' . $this->shopkit_settings['product_image_hover_style'] ;
                } else {
                    $new_classes .= ' image-hover-icon';
                }
                if ( isset( $this->shopkit_settings['product_image_hover_position'] ) && $this->shopkit_settings['product_image_hover_position'] ) {
                    $new_classes .=  ' hover-' . $this->shopkit_settings['product_image_hover_position'];
                } else {
                    $new_classes .= ' hover-right';
                }
                $classes = array_merge( $classes, [ $new_classes ] );
            //}
            
            // Add shimmer effect for skeleton
            $classes = array_merge( array( 'shimmer' ), $classes );

            return $classes;
        }

        public function quickview_button() {
            echo do_shortcode('[woosq]');
        }

        public function compare_button() {
            echo do_shortcode('[woosc]');
        }

        public function wishlist_button() {
            echo do_shortcode('[woosw]');
        }

        /**
         * Set Cross sells limit.
         *
         * @param string $limit the current product limit.
         */
        public function cross_sell_limit( $limit ) {
            return ( isset( $this->shopkit_settings['cross_sell_limit'] ) && !empty($this->shopkit_settings['cross_sell_limit']) ) ? $this->shopkit_settings['cross_sell_limit'] : 5;
        }
        /**
         * Set Cross sells columns.
         *
         * @param string $columns the current column count.
         */
        public function cross_sell_columns( $columns ) {
            return ( isset( $this->shopkit_settings['cross_sell_columns'] ) && !empty($this->shopkit_settings['cross_sell_columns']) ) ? $this->shopkit_settings['cross_sell_columns'] : 5;
        }
        
        /**
         * Shop products on not found page
         */
        public function tmcore_show_product_on_not_found() {
            ?>
            <div class="search-featured-wrapper">
                <h6><?php esc_html_e( 'Based on what you were looking for, you might like:', 'templatemela-core' ); ?></h6>
                <br/>
                <?php echo do_shortcode( '[products limit="8" columns="4" visibility="featured" orderby="rand"]'); ?>
            </div>
            <?php
        }
        

    }
    return new TemplateMelaCore_WooCommerce();
endif;
