<?php
/**
 * TemplateMela functions.
 *
 * @since 1.0.0
 * @package TemplateMela Core
 */

use function Base\webapp;

if (!defined('ABSPATH')) {
    exit;
} 



/**
 * Locate Template
 *
 * @access public
 * @since 1.0.0
 * @return void
 */

function tmcore_locate_output_template( $template_name, $template_args = array(), $default_path = '' ) {

    $template_path = 'templatemela/';
    $default_path  = TMCORE_PATH . 'includes/templates/';

    // Look within passed path within the theme - this is priority.
    $template = locate_template(
        array(
            trailingslashit( $template_path ) . $template_name,
            $template_name,
        )
    );
    // Get default template/.
    if ( ! $template ) {
        $template = $default_path . $template_name;
    }

    // Allow 3rd party plugin filter template file from their plugin.
    $filter_template = apply_filters( 'templatemela_get_template', $template, $template_name, $template_path, $default_path );

    if ( $filter_template !== $template ) {
        if ( ! file_exists( $filter_template ) ) {
            return;
        }
        $template = $filter_template;
    }

    include $template;
}




/**
 * Call a shortcode function by tag name.
 *
 * @param string $tag The shortcode whose function to call.
 * @param array $atts The attributes to pass to the shortcode function. Optional.
 * @param array $content The shortcode's content. Default is null (none).
 *
 * @return string|bool False on failure, the result of the shortcode on success.
 * @since  1.4.6
 *
 */
function tmcore_do_shortcode( $tag, array $atts = array(), $content = null ) {
	global $shortcode_tags;

	if ( ! isset( $shortcode_tags[ $tag ] ) ) {
		return false;
	}

	return call_user_func( $shortcode_tags[ $tag ], $atts, $content, $tag );
}


/**
 * Get darken color
 *
 * @access public
 * @since 1.0.0
 */
function tmcore_darken_color($rgb, $darker = 1.1) {

    $hash = (strpos($rgb, '#') !== false) ? '#' : '';
    $rgb  = (strlen($rgb) == 7) ? str_replace('#', '', $rgb) : ((strlen($rgb) == 6) ? $rgb : false);
    if (strlen($rgb) != 6) return $hash . '000000';
    $darker = ($darker > 1) ? $darker : 1;

    list($R16, $G16, $B16) = str_split($rgb, 2);

    $R = sprintf("%02X", floor(hexdec($R16) / $darker));
    $G = sprintf("%02X", floor(hexdec($G16) / $darker));
    $B = sprintf("%02X", floor(hexdec($B16) / $darker));

    return $hash . $R . $G . $B;
}


function tmcore_the_attachment_by_id( $args = array() ) {

    $id = $size = $width = $height = $crop = '';

    $id = $args['id'];
    $size = $args['size'];
    $attachment = wp_get_attachment_image( $id, $size );

    echo "{$attachment}";
}
function tmcore_get_attachment_url_by_id( $args = array() ) {
    $id = $size = $width = $height = $crop = '';

    $defaults = array(
        'id'      => '',
        'size'    => 'full',
        'width'   => '',
        'height'  => '',
        'crop'    => true,
        'details' => false,
    );

    $args = wp_parse_args( $args, $defaults );
    extract( $args );

    if ( $id === '' ) {
        return '';
    }

    if ( $details === false ) {
        $url           = wp_get_attachment_image_url( $id, 'full' );
        $image_cropped = tmcore_get_image_cropped_url( $url, $args );

        return $image_cropped[0];
    } else {
        $image_full = tmcore_get_attachment_info( $id );
        $url        = $image_full['src'];

        $image_cropped = tmcore_get_image_cropped_url( $url, $args );

        $full_details                  = $image_full;
        $full_details['cropped_image'] = $image_cropped[0];

        return $full_details;
    }
}

/**
 * @param string $url  Original image url.
 * @param array  $args Array attributes.
 *
 * @return array|bool|string
 */
function tmcore_get_image_cropped_url( $url, $args = array() ) {
    extract( $args );
    if ( $url === false ) {
        return array( 0 => '' );
    }

    if ( $size === 'full' ) {
        return array( 0 => $url );
    }

    if ( $size !== 'custom' && ! preg_match( '/(\d+)x(\d+)/', $size ) ) {
        $attachment_url = wp_get_attachment_image_url( $args['id'], $size );

        if ( ! $attachment_url ) {
            return array( 0 => $url );
        } else {
            return array( 0 => $attachment_url );
        }
    }

    if ( $size !== 'custom' ) {
        $_sizes = explode( 'x', $size );
        $width  = $_sizes[0];
        $height = $_sizes[1];
    } else {
        if ( $width === '' ) {
            $width = 9999;
        }

        if ( $height === '' ) {
            $height = 9999;
        }
    }

    $width  = (int) $width;
    $height = (int) $height;

    if ( $width === 9999 || $height === 9999 ) {
        $crop = false;
    }

    if ( $width !== '' && $height !== '' && function_exists( 'aq_resize' ) ) {
        $crop_image = aq_resize( $url, $width, $height, $crop, false );

        if ( ! empty( $crop_image ) && is_array( $crop_image ) && ! empty( $crop_image[0] ) ) {
            return $crop_image;
        }
    }

    return array( 0 => $url );
}

function tmcore_get_attachment_info( $attachment_id ) {
    if ( empty( $attachment_id ) ) {
        return false;
    }

    $attachment = get_post( $attachment_id );

    $attachment_types = [
        'attachment',
        'vdl_remote_image', // Compatible with Woosa - vidaXL dropshipping for WooCommerce.
    ];

    if ( ! $attachment instanceof WP_Post || ! in_array( $attachment->post_type, $attachment_types ) ) {
        return false;
    }

    $image_src = wp_get_attachment_image_src( $attachment_id, 'full', false );

    if ( empty( $image_src[0] ) ) {
        return false;
    }

    list ( $attachment_url, $width, $height ) = $image_src;

    $alt = get_post_meta( $attachment->ID, '_wp_attachment_image_alt', true );

    if ( '' === $alt ) {
        $alt = $attachment->post_title;
    }

    return array(
        'alt'         => $alt,
        'caption'     => $attachment->post_excerpt,
        'description' => $attachment->post_content,
        'href'        => get_permalink( $attachment->ID ),
        'src'         => $attachment_url,
        'title'       => $attachment->post_title,
    );
}



/**
 * @param array $attributes
 *
 * @return string
 */
function tmcore_convert_array_html_attributes_to_string( $attributes ) {
    $attr_str = '';

    foreach ( $attributes as $name => $value ) {
        switch ( $name ) {
            case 'href':
            case 'src':
            case 'data-src':
                $attr_str .= ' ' . $name . '="' . esc_url( $value ) . '"';
                break;
            case 'class':
                $value = is_array( $value ) ? implode( ' ', $value ) : $value;

                $attr_str .= ' ' . $name . '="' . esc_attr( $value ) . '"';
                break;
            default:
                $value = is_array( $value ) ? wp_json_encode( $value ) : $value;

                $attr_str .= ' ' . $name . '="' . esc_attr( $value ) . '"';
                break;
        }
    }

    return $attr_str;
}

function grid_args_to_html_attr( $args ) {
    $style = grid_args_to_html_style( $args );

    if ( ! empty( $style ) ) {
        return ' style="' . $style . '"';
    }

    return '';
}

function grid_args_to_html_style( $args ) {
    $style = '';

    $selectors_dictionary = [
        'columnsWidescreen'  => '--grid-columns-widescreen',
        'columns'            => '--grid-columns-desktop',
        'columnsLaptop'      => '--grid-columns-laptop',
        'columnsTabletExtra' => '--grid-columns-tablet-extra',
        'columnsTablet'      => '--grid-columns-tablet',
        'columnsMobileExtra' => '--grid-columns-mobile-extra',
        'columnsMobile'      => '--grid-columns-mobile',
        'gutterWidescreen'   => '--grid-gutter-widescreen',
        'gutter'             => '--grid-gutter-desktop',
        'gutterLaptop'       => '--grid-gutter-laptop',
        'gutterTabletExtra'  => '--grid-gutter-tablet-extra',
        'gutterTablet'       => '--grid-gutter-tablet',
        'gutterMobileExtra'  => '--grid-gutter-mobile-extra',
        'gutterMobile'       => '--grid-gutter-mobile',
    ];

    foreach ( $args as $name => $value ) {
        if ( isset( $selectors_dictionary[ $name ] ) ) {
            $css_var = $selectors_dictionary[ $name ];
            $style   .= "$css_var: $value;";
        }
    }

    return $style;
}


function tmcore_get_product_highlight_filter_options() {
    return apply_filters( 'tmcore_wp_widget_product_highlight_filter_options', [
        ''             => esc_html__( 'All Products', 'templatemela-core' ),
        'best_selling' => esc_html__( 'Best Seller', 'templatemela-core' ),
        'new_arrivals' => esc_html__( 'New Arrivals', 'templatemela-core' ),
        'on_sale'      => esc_html__( 'Sale', 'templatemela-core' ),
        'featured'     => esc_html__( 'Hot Items', 'templatemela-core' ),
    ] );
}



function tmcore_get_shop_active_filters_url( $filters = array(), $link = '' ) {
    if ( empty( $link ) ) {
        $link = tmcore_get_shop_base_url();
    }

    if ( empty( $filters ) ) {
        $filters = $_GET;
    }

    // Min/Max.
    if ( isset( $filters['min_price'] ) ) {
        $link = add_query_arg( 'min_price', wc_clean( wp_unslash( $filters['min_price'] ) ), $link );
    }

    if ( isset( $filters['max_price'] ) ) {
        $link = add_query_arg( 'max_price', wc_clean( wp_unslash( $filters['max_price'] ) ), $link );
    }

    // Order by.
    if ( isset( $filters['orderby'] ) ) {
        $link = add_query_arg( 'orderby', wc_clean( wp_unslash( $filters['orderby'] ) ), $link );
    }

    /**
     * Search Arg.
     * To support quote characters, first they are decoded from &quot; entities, then URL encoded.
     */
    if ( get_search_query() ) {
        $link = add_query_arg( 's', rawurlencode( wp_specialchars_decode( get_search_query() ) ), $link );
    }

    // Post Type Arg.
    if ( isset( $filters['post_type'] ) ) {
        $link = add_query_arg( 'post_type', wc_clean( wp_unslash( $filters['post_type'] ) ), $link );

        // Prevent post type and page id when pretty permalinks are disabled.
        if ( is_shop() ) {
            $link = remove_query_arg( 'page_id', $link );
        }
    }

    // Min Rating Arg.
    if ( isset( $filters['rating_filter'] ) ) {
        $link = add_query_arg( 'rating_filter', wc_clean( wp_unslash( $filters['rating_filter'] ) ), $link );
    }

    if ( ! empty( $filters['highlight_filter'] ) ) {
        $link = add_query_arg( 'highlight_filter', wc_clean( wp_unslash( $filters['highlight_filter'] ) ), $link );
    }

    if ( ! empty( $filters['filter_product_cat'] ) ) {
        $link = add_query_arg( 'filter_product_cat', wc_clean( wp_unslash( $filters['filter_product_cat'] ) ), $link );
    }

    if ( ! empty( $filters['filter_product_brand'] ) ) {
        $link = add_query_arg( 'filter_product_brand', wc_clean( wp_unslash( $filters['filter_product_brand'] ) ), $link );
    }

    // All current filters.
    if ( $_chosen_attributes = \TemplateMelaCore\Woo\Product_Query::get_layered_nav_chosen_attributes() ) { // phpcs:ignore Squiz.PHP.DisallowMultipleAssignments.Found, WordPress.CodeAnalysis.AssignmentInCondition.Found
        foreach ( $_chosen_attributes as $name => $data ) {
            $filter_name = wc_attribute_taxonomy_slug( $name );

            if ( ! empty( $data['terms'] ) ) {
                $link = add_query_arg( 'filter_' . $filter_name, implode( ',', $data['terms'] ), $link );
            }
            if ( 'or' === $data['query_type'] ) {
                $link = add_query_arg( 'query_type_' . $filter_name, 'or', $link );
            }
        }
    }

    return $link;
}


function tmcore_get_shop_base_url() {
    if ( defined( 'SHOP_IS_ON_FRONT' ) ) {
        $link = home_url();
    } elseif ( is_shop() ) {
        $link = get_permalink( wc_get_page_id( 'shop' ) );
    } elseif ( is_product_category() ) {
        $link = get_term_link( get_query_var( 'product_cat' ), 'product_cat' );
    } elseif ( is_product_tag() ) {
        $link = get_term_link( get_query_var( 'product_tag' ), 'product_tag' );
    } else {
        $queried_object = get_queried_object();
        $link           = get_term_link( $queried_object->slug, $queried_object->taxonomy );
    }

    return $link;
}


function tmcore_image_placeholder( $width, $height ) {
    echo '<img src="https://via.placeholder.com/' . $width . 'x' . $height . '?text=' . esc_attr__( 'No+Image', 'templatemela-core' ) . '" alt="' . esc_attr__( 'Thumbnail', 'templatemela-core' ) . '"/>';
}


function tmcore_render_product_button( $args ) {
    $defaults = [
        'wrapper'            => true,
        'wrapper_class'      => '',
        'wrapper_attributes' => [],
        'text'               => '',
        'link'               => [
            'url'         => '',
            'is_external' => false,
            'nofollow'    => false,
        ],
        'style'              => 'flat',
        'size'               => 'nm',
        'full_wide'          => false,
        'icon'               => '',
        'icon_align'         => 'left',
        'extra_class'        => '',
        'class'              => 'tm-button',
        'id'                 => '',
        'attributes'         => [],
        'echo'               => true,
    ];

    $args = wp_parse_args( $args, $defaults );
    extract( $args );

    $button_attrs = wp_parse_args( [], $attributes );

    $button_classes   = [ $class ];
    $button_classes[] = 'style-' . $style;
    $button_classes[] = 'tm-button-' . $size;

    if ( $full_wide ) {
        $button_classes[] = 'tm-button-full-wide';
    }

    if ( ! empty( $extra_class ) ) {
        $button_classes[] = $extra_class;
    }

    if ( ! empty( $icon ) ) {
        $button_classes[] = 'icon-' . $icon_align;
    }

    $button_attrs['class'] = implode( ' ', $button_classes );

    if ( ! empty( $id ) ) {
        $button_attrs['id'] = $id;
    }

    $button_tag = 'div';

    if ( ! empty( $link['url'] ) ) {
        $button_tag = 'a';

        $button_attrs['href'] = $link['url'];

        if ( ! empty( $link['is_external'] ) ) {
            $button_attrs['target'] = '_blank';
        }

        if ( ! empty( $link['nofollow'] ) ) {
            $button_attrs['rel'] = $link['nofollow'];
        }
    }

    $attributes_str = '';

    if ( ! empty( $button_attrs ) ) {
        foreach ( $button_attrs as $attribute => $value ) {
            $attributes_str .= ' ' . $attribute . '="' . esc_attr( $value ) . '"';
        }
    }

    $wrapper_classes = 'tm-button-wrapper';
    if ( ! empty( $wrapper_class ) ) {
        $wrapper_classes .= " $wrapper_class";
    }

    $wrapper_attributes_str = '';
    if ( ! empty( $wrapper_attributes ) ) {
        foreach ( $wrapper_attributes as $attribute => $value ) {
            $wrapper_attributes_str .= ' ' . $attribute . '="' . esc_attr( $value ) . '"';
        }
    }

    ob_start();
    ?>
    <?php printf( '<%1$s %2$s>', $button_tag, $attributes_str ); ?>
    <div class="button-content-wrapper">

        <?php if ( ! empty( $icon ) && 'left' === $icon_align ): ?>
            <span class="button-icon"><i class="<?php echo esc_attr( $icon ); ?>"></i></span>
        <?php endif; ?>

        <?php if ( ! empty( $text ) ): ?>
            <span class="button-text"><?php echo esc_html( $text ); ?></span>
        <?php endif; ?>

        <?php if ( ! empty( $icon ) && 'right' === $icon_align ): ?>
            <span class="button-icon"><i class="<?php echo esc_attr( $icon ); ?>"></i></span>
        <?php endif; ?>

        <?php if ( $style === 'bottom-line-winding' ): ?>
            <span class="line-winding">
                    <svg width="42" height="6" viewBox="0 0 42 6" fill="none"
                         xmlns="http://www.w3.org/2000/svg">
                        <path fill-rule="evenodd" clip-rule="evenodd"
                              d="M0.29067 2.60873C1.30745 1.43136 2.72825 0.72982 4.24924 0.700808C5.77022 0.671796 7.21674 1.31864 8.27768 2.45638C8.97697 3.20628 9.88872 3.59378 10.8053 3.5763C11.7218 3.55882 12.6181 3.13683 13.2883 2.36081C14.3051 1.18344 15.7259 0.481897 17.2469 0.452885C18.7679 0.423873 20.2144 1.07072 21.2753 2.20846C21.9746 2.95836 22.8864 3.34586 23.8029 3.32838C24.7182 3.31092 25.6133 2.89009 26.2831 2.11613C26.2841 2.11505 26.285 2.11396 26.2859 2.11288C27.3027 0.935512 28.7235 0.233974 30.2445 0.204962C31.7655 0.17595 33.212 0.822796 34.2729 1.96053C34.9722 2.71044 35.884 3.09794 36.8005 3.08045C37.7171 3.06297 38.6134 2.64098 39.2836 1.86496C39.6445 1.44697 40.276 1.40075 40.694 1.76173C41.112 2.12271 41.1582 2.75418 40.7972 3.17217C39.7804 4.34954 38.3597 5.05108 36.8387 5.08009C35.3177 5.1091 33.8712 4.46226 32.8102 3.32452C32.1109 2.57462 31.1992 2.18712 30.2826 2.2046C29.3674 2.22206 28.4723 2.64289 27.8024 3.41684C27.8015 3.41793 27.8005 3.41901 27.7996 3.42009C26.7828 4.59746 25.362 5.299 23.841 5.32801C22.3201 5.35703 20.8735 4.71018 19.8126 3.57244C19.1133 2.82254 18.2016 2.43504 17.285 2.45252C16.3685 2.47 15.4722 2.89199 14.802 3.66802C13.7852 4.84539 12.3644 5.54693 10.8434 5.57594C9.32242 5.60495 7.8759 4.9581 6.81496 3.82037C6.11568 3.07046 5.20392 2.68296 4.28738 2.70044C3.37083 2.71793 2.47452 3.13992 1.80434 3.91594C1.44336 4.33393 0.811887 4.38015 0.393899 4.01917C-0.0240897 3.65819 -0.0703068 3.02672 0.29067 2.60873Z"
                              fill="#E8C8B3"/>
                    </svg>
                </span>
        <?php endif; ?>
    </div>
    <?php printf( '</%1$s>', $button_tag ); ?>
    <?php
    $output = ob_get_clean();

    if ( $wrapper ) {
        $output = sprintf( '<div class="%1$s" %2$s>%3$s</div>', esc_attr( $wrapper_classes ), $wrapper_attributes_str, $output );
    }

    if ( $echo ) {
        echo '' . $output;
    }

    return $output;
}



function tmcore_render_product_rating( $rating = 5, $args = array() ) {
    $default = [
        'style'         => '01',
        'wrapper_class' => '',
        'echo'          => true,
    ];

    $args = wp_parse_args( $args, $default );

    $el_classes = 'tm-star-rating style-' . $args['style'];
    if ( ! empty( $args['wrapper_class'] ) ) {
        $el_classes .= " {$args['wrapper_class']}";
    }

    $full_stars = intval( $rating );

    $star_full_icon = '<svg xmlns="http://www.w3.org/2000/svg" x="0px" y="0px" viewBox="0 0 512 512" xml:space="preserve" class="tm-star-full"><path fill="#000000" d="M512,197.816L325.961 185.585 255.898 9.569 185.835 185.585 0 197.816 142.534 318.842 95.762 502.431 255.898 401.21 416.035 502.431 369.263 318.842z"></path></svg>';

    $star_empty_icon = '<svg xmlns="http://www.w3.org/2000/svg" x="0px" y="0px" viewBox="0 0 512 512" xml:space="preserve" class="tm-star-empty"><path fill="#000000 " d="M512,197.819l-185.933-12.228L256,9.571l-70.067,176.021L0,197.82l142.658,120.93L95.856,502.429L256,401.214 l160.144,101.215l-46.8-183.671L512,197.819z M256,365.724l-112.464,71.08l32.827-128.831L75.829,222.888l130.971-8.603 L256,90.687l49.2,123.599l131.124,8.602l-100.689,85.077l32.829,128.839L256,365.724z"></path></svg>';

    $template = '';
    $template .= str_repeat( $star_full_icon, $full_stars );

    $half_star = floatval( $rating ) - $full_stars;

    if ( $half_star != 0 ) {
        $star_half_icon = '<svg xmlns="http://www.w3.org/2000/svg" x="0px" y="0px" viewBox="0 0 510 510" xml:space="preserve" class="tm-star-half"><path fill="#000000" d="M510,197.472l-183.37-15.734L255,12.75l-71.629,168.988L0,197.472l0,0l0,0l139.103,120.539L97.41,497.25L255,402.186l0,0l157.59,95.039l-41.692-179.239L510,197.472z M255,354.348V117.172l43.605,102.918l111.689,9.588l-84.711,73.389l25.398,109.166L255,354.348z"></path></svg>';
        $template       .= $star_half_icon;
    }

    $empty_stars = intval( 5 - $rating );
    $template    .= str_repeat( $star_empty_icon, $empty_stars );

    $template = '<div class="' . esc_attr( $el_classes ) . '">' . $template . '</div>';

    if ( $args['echo'] ) {
        echo '' . $template;
    }

    return $template;
}


/**
 * Get ids of products that largest total sales.
 *
 * @return array|mixed
 */
function tmcore_get_product_ids_best_selling() {
    
    if ( empty( $best_selling_products ) ) {
        $good_selling_products  = tmcore_get_product_ids_good_selling();
        $number                 = intval( TemplateMelaCore_Settings::get_setting_option( 'shop_best_selling_list_number' ) );
        $best_selling_products  = array_slice( $good_selling_products, 0, $number );
        return $best_selling_products;
    }
    
}

function tmcore_get_product_ids_good_selling() {
    $transient_name = 'tmcore-product-ids-good-selling';
    //$products = get_transient( $transient_name );
    $products = false;
    if ( false === $products ) {
        global $wpdb;
        $products = [];

        $sql = "SELECT product.ID FROM $wpdb->posts as product 
                INNER JOIN $wpdb->postmeta ON ( product.ID = {$wpdb->postmeta}.post_id ) 
                WHERE {$wpdb->postmeta}.meta_key = 'total_sales' AND product.post_type = 'product' AND product.post_status = 'publish' 
                GROUP BY product.ID ORDER BY {$wpdb->postmeta}.meta_value+0 DESC";

        $results = $wpdb->get_col( $sql );

        if ( ! empty( $results ) ) {
            $products = $results;
        }

        //set_transient( $transient_name, $products, DAY_IN_SECONDS );
    }

    return $products;
}


/**
 * Get ids of products that new arrivals.
 *
 * @return array|mixed
 */
function tmcore_get_product_ids_new_arrivals() {
    $transient_name = 'tmcore-product-ids-new-arrivals';

    //$products = get_transient( $transient_name );
    $products = false;
    if ( false === $products ) {
        global $wpdb;
        $day        = 7;//TemplateMelaCore_Settings::get_setting_option( 'shop_badge_new_range' );
        $date_range = strtotime( "-{$day} day" );
        $products   = [];

        $date_query = date( 'Y-m-d H:i:s', $date_range );

        $sql = "SELECT product.ID FROM {$wpdb->posts} AS product
                WHERE 1=1 AND ( product.post_date > %s )
                AND product.post_type = 'product' AND product.post_status = 'publish' 
                ORDER BY product.post_date DESC";

        $sql     = $wpdb->prepare( $sql, $date_query );
        $results = $wpdb->get_col( $sql );

        if ( ! empty( $results ) ) {
            $products = $results;
        }

        //set_transient( $transient_name, $products, DAY_IN_SECONDS );
    }

    return $products;
}


function tmcore_get_remove_active_filter_links( $filters ) {
    $filter_link = tmcore_get_shop_active_filters_url( $filters );

    $clear_links = [];

    foreach ( $filters as $filter_name => $value ) {
        $taxonomy_name = 0 === strpos( $filter_name, 'filter_' ) ? wc_sanitize_taxonomy_name( str_replace( 'filter_', '', $filter_name ) ) : '';

        $attribute_name = wc_attribute_taxonomy_name( $taxonomy_name );
        $attribute_id   = ! empty( $attribute_name ) ? wc_attribute_taxonomy_id_by_name( $attribute_name ) : 0;

        // This is taxonomy filter like category, tag, brand...
        if ( ! empty( $taxonomy_name ) && taxonomy_exists( $taxonomy_name ) ) {
            $taxonomy = get_taxonomy( $taxonomy_name );

            $filter_terms = ! empty( $value ) ? explode( ',', wc_clean( wp_unslash( $value ) ) ) : array();

            if ( empty( $filter_terms ) ) {
                continue;
            }

            foreach ( $filter_terms as $key => $term_id ) {
                $clear_link = $filter_link;
                $clear_link = remove_query_arg( $filter_name, $clear_link );

                $term = get_term_by( 'id', $term_id, $taxonomy_name );

                if ( empty( $term ) ) {
                    continue;
                }

                $clone_terms = $filter_terms;
                unset( $clone_terms[ $key ] );

                if ( empty( $clone_terms ) ) {
                    $clear_link = remove_query_arg( $filter_name, $clear_link );
                } else {
                    // Re add.
                    $clear_link = add_query_arg( $filter_name, implode( ',', $clone_terms ), $clear_link );
                }

                $tooltip_text = isset( $taxonomy->labels->singular_name ) ? $taxonomy->labels->singular_name : __( 'Filter', 'templatemela-core' );

                $clear_links[] = [
                    'url'     => $clear_link,
                    'text'    => $term->name,
                    'tooltip' => sprintf( __( 'Remove This %s', 'templatemela-core' ), $tooltip_text ),
                    'class'   => 'remove-filter-link',
                ];
            }
        } elseif ( $attribute_id && taxonomy_exists( $attribute_name ) ) { // This is attribute filter like color, size...
            $filter_terms = ! empty( $value ) ? explode( ',', wc_clean( wp_unslash( $value ) ) ) : array();

            if ( empty( $filter_terms ) ) {
                continue;
            }

            $attribute_info = wc_get_attribute( $attribute_id );

            foreach ( $filter_terms as $key => $term_slug ) {
                $clear_link = $filter_link;
                $clear_link = remove_query_arg( $filter_name, $clear_link );

                $term = get_term_by( 'slug', $term_slug, $attribute_name );

                if ( empty( $term ) ) {
                    continue;
                }

                $clone_terms = $filter_terms;
                unset( $clone_terms[ $key ] );

                if ( empty( $clone_terms ) ) {
                    $clear_link = remove_query_arg( $filter_name, $clear_link );
                } else {
                    // Re add.
                    $clear_link = add_query_arg( $filter_name, implode( ',', $clone_terms ), $clear_link );
                }

                $clear_links[] = [
                    'url'     => $clear_link,
                    'text'    => $term->name,
                    'tooltip' => sprintf( __( 'Remove This %s', 'templatemela-core' ), $attribute_info->name ),
                    'class'   => 'remove-filter-link',
                ];
            }
        } elseif ( 'rating_filter' === $filter_name ) {
            $filter_values = ! empty( $value ) ? explode( ',', wc_clean( wp_unslash( $value ) ) ) : array();

            if ( empty( $filter_values ) ) {
                continue;
            }

            foreach ( $filter_values as $key => $filter_value ) {
                $clear_link = $filter_link;
                $clear_link = remove_query_arg( $filter_name, $clear_link );

                $clone_values = $filter_values;
                unset( $clone_values[ $key ] );

                if ( empty( $clone_values ) ) {
                    $clear_link = remove_query_arg( $filter_name, $clear_link );
                } else {
                    // Re add.
                    $clear_link = add_query_arg( $filter_name, implode( ',', $clone_values ), $clear_link );
                }

                $clear_links[] = [
                    'url'     => $clear_link,
                    'text'    => sprintf( _n( '%s star', '%s stars', $filter_value, 'templatemela-core' ), $filter_value ),
                    'tooltip' => sprintf( __( 'Remove This %s', 'templatemela-core' ), __( 'Rating', 'templatemela-core' ) ),
                    'class'   => 'remove-filter-link',
                ];
            }
        } elseif ( 'highlight_filter' === $filter_name ) {
            $clear_link        = $filter_link;
            $clear_link        = remove_query_arg( $filter_name, $clear_link );
            $highlight_options = tmcore_get_product_highlight_filter_options();

            $clear_link_text = isset( $highlight_options[ $value ] ) ? $highlight_options[ $value ] : $value;

            $clear_links[] = [
                'url'     => $clear_link,
                'text'    => $clear_link_text,
                'tooltip' => sprintf( __( 'Remove This %s', 'templatemela-core' ), __( 'Highlight', 'templatemela-core' ) ),
                'class'   => 'remove-filter-link',
            ];
        }
    }

    if ( isset( $filters['min_price'] ) && isset( $filters['max_price'] ) ) {
        $clear_link = $filter_link;
        $clear_link = remove_query_arg( 'min_price', $clear_link );
        $clear_link = remove_query_arg( 'max_price', $clear_link );

        $clear_links[] = [
            'url'     => $clear_link,
            'text'    => wc_price( $filters['min_price'] ) . ' - ' . wc_price( $filters['max_price'] ),
            'tooltip' => sprintf( __( 'Remove This %s', 'templatemela-core' ), __( 'Price', 'templatemela-core' ) ),
            'class'   => 'remove-filter-link',
        ];
    }

    if ( webapp()->option( 'product_archive_shop_filter_active_remove_all' ) ) {
        if ( ! empty( $filters['filtering'] ) || ! empty( $clear_links ) ) {
            $clear_links[] = [
                'url'   => tmcore_get_shop_base_url(),
                'text'  => esc_html__( 'Clear All', 'templatemela-core' ),
                'class' => 'remove-all-filters-link',
            ];
        }
    }

    $output = '<div class="active-filters-list">';

    foreach ( $clear_links as $clear_link ) {
        $base_link_class = 'js-product-filter-link';

        if ( ! empty( $clear_link['class'] ) ) {
            $base_link_class .= " {$clear_link['class']}";
        }

        if ( ! empty( $clear_link['tooltip'] ) ) {
            $base_link_class .= ' hint--bounce hint--top';
        }

        $tooltip_text = ! empty( $clear_link['tooltip'] ) ? $clear_link['tooltip'] : esc_html__( 'Remove This Filter', 'templatemela-core' );

        $output .= sprintf( '<a href="%1$s" class="%2$s" aria-label="%3$s"><div class="filter-link-text">%4$s</div></a>', $clear_link['url'], $base_link_class, $tooltip_text, $clear_link['text'] );
    }

    $output .= '</div>';

    return $output;
}





// POSTS FUNCTIONS

if (!function_exists('tmcore_post_header')) {
    /**
     * Display the post header with a link to the single post
     *
     * @since 1.0.0
     */
    function tmcore_post_header() {
        ?>
        <header class="entry-header">
            <?php

            /**
             * Functions hooked in to tmcore_post_header_before action.
             */
            do_action('tmcore_post_header_before');
            ?>
            <div class="entry-meta">
                <?php
                    tmcore_post_meta();
                ?>
            </div>
            <?php
            if (is_single()) {
                if (!(function_exists( 'bcn_display' ) || class_exists( 'WooCommerce' ))) {
                    the_title('<h1 class="entry-title">', '</h1>');
                }
            } else {
                the_title(sprintf('<h2 class="alpha entry-title"><a href="%s" rel="bookmark">', esc_url(get_permalink())), '</a></h2>');
            }

            do_action('tmcore_post_header_after');
            ?>
        </header><!-- .entry-header -->
        <?php
    }
}

if (!function_exists('tmcore_post_content')) {
    /**
     * Display the post content with a link to the single post
     *
     * @since 1.0.0
     */
    function tmcore_post_content() {
        ?>
        <div class="entry-content">
            <?php

            /**
             * Functions hooked in to tmcore_post_content_before action.
             *
             */
            do_action('tmcore_post_content_before');


            the_content(
                sprintf(
                /* translators: %s: post title */
                    __('Read More %s', 'templatemela-core'),
                    '<span class="screen-reader-text">' . get_the_title() . '</span>'
                )
            );

            /**
             * Functions hooked in to tmcore_post_content_after action.
             *
             */
            do_action('tmcore_post_content_after');

            wp_link_pages(
                array(
                    'before' => '<div class="page-links">' . esc_html__('Pages:', 'templatemela-core'),
                    'after'  => '</div>',
                )
            );
            ?>
        </div><!-- .entry-content -->
        <?php
    }
}

if (!function_exists('tmcore_post_meta')) {
    /**
     * Display the post meta
     *
     * @since 1.0.0
     */
    function tmcore_post_meta() {
        if ('post' !== get_post_type()) {
            return;
        }

        // Posted on.
        $time_string = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';

        if (get_the_time('U') !== get_the_modified_time('U')) {
            $time_string = '<time class="entry-date published" datetime="%1$s">%2$s</time><time class="updated" datetime="%3$s">%4$s</time>';
        }

        $time_string = sprintf(
            $time_string,
            esc_attr(get_the_date('c')),
            esc_html(get_the_date()),
            esc_attr(get_the_modified_date('c')),
            esc_html(get_the_modified_date())
        );

        $posted_on = '<span class="posted-on">' . sprintf('<a href="%1$s" rel="bookmark">%2$s</a>', esc_url(get_permalink()), $time_string) . '</span>';

        // Author.
        $author = sprintf(
            '<span class="post-author">%1$s&nbsp;<a href="%2$s" class="url fn" rel="author">%3$s</a></span>',
            __('by', 'templatemela-core'),
            esc_url(get_author_posts_url(get_the_author_meta('ID'))),
            esc_html(get_the_author())
        );


        echo wp_kses(
            sprintf('%1$s %2$s', $posted_on, $author), array(
                'span' => array(
                    'class' => array(),
                ),
                'a'    => array(
                    'href'  => array(),
                    'title' => array(),
                    'rel'   => array(),
                ),
                'time' => array(
                    'datetime' => array(),
                    'class'    => array(),
                ),
            )
        );
    }
}

if (!function_exists('tmcore_edit_post_link')) {
    /**
     * Display the edit link
     *
     * @since 2.5.0
     */
    function tmcore_edit_post_link() {
        edit_post_link(
            sprintf(
                wp_kses(
                    /* translators: %s: Name of current post. Only visible to screen readers */
                    __('Edit <span class="screen-reader-text">%s</span>', 'templatemela-core'),
                    array(
                        'span' => array(
                            'class' => array(),
                        ),
                    )
                ),
                get_the_title()
            ),
            '<div class="edit-link">',
            '</div>'
        );
    }
}

if (!function_exists('tmcore_post_taxonomy')) {
    /**
     * Display the post taxonomies
     *
     * @since 2.4.0
     */
    function tmcore_post_taxonomy() {
        /* translators: used between list items, there is a space after the comma */
        $categories_list = get_the_category_list(__(', ', 'templatemela-core'));

        /* translators: used between list items, there is a space after the comma */
        $tags_list = get_the_tag_list('', __(', ', 'templatemela-core'));
        ?>
        <aside class="entry-taxonomy">
            <?php if ($categories_list) : ?>
                <div class="cat-links">
                    <strong><?php echo esc_html(_n('Category:', 'Categories:', count(get_the_category()), 'templatemela-core')); ?></strong><?php echo get_the_category_list(esc_html__(', ', 'templatemela-core')); ?>
                </div>
            <?php endif; ?>

            <?php if ($tags_list) : ?>
                <div class="tags-links">
                    <strong><?php echo esc_html(_n('Tag:', 'Tags:', count(get_the_tags()), 'templatemela-core')); ?></strong><?php echo get_the_tag_list('', esc_html__(', ', 'templatemela-core')); ?>
                </div>
            <?php endif; ?>
        </aside>

        <?php
    }
}

if (!function_exists('tmcore_paging_nav')) {
    /**
     * Display navigation to next/previous set of posts when applicable.
     */
    function tmcore_paging_nav() {
        global $wp_query;

        $args = array(
            'type'      => 'list',
            'next_text' => _x('Next', 'Next post', 'templatemela-core'),
            'prev_text' => _x('Previous', 'Previous post', 'templatemela-core'),
        );

        the_posts_pagination($args);
    }
}

if (!function_exists('tmcore_post_nav')) {
    /**
     * Display navigation to next/previous post when applicable.
     */
    function tmcore_post_nav() {
        $args = array(
            'next_text' => '<span class="screen-reader-text">' . esc_html__('Next post:', 'templatemela-core') . ' </span>%title',
            'prev_text' => '<span class="screen-reader-text">' . esc_html__('Previous post:', 'templatemela-core') . ' </span>%title',
        );
        the_post_navigation($args);
    }
}

if (!function_exists('tmcore_posted_on')) {
    /**
     * Prints HTML with meta information for the current post-date/time and author.
     *
     * @deprecated 2.4.0
     */
    function tmcore_posted_on() {
        _deprecated_function('tmcore_posted_on', '2.4.0');
    }
}


if (!function_exists('tmcore_post_thumbnail')) {
    /**
     * Display post thumbnail
     *
     * @param string $size the post thumbnail size.
     *
     * @uses has_post_thumbnail()
     * @uses the_post_thumbnail
     * @var $size thumbnail size. thumbnail|medium|large|full|$custom
     * @since 1.5.0
     */
    function tmcore_post_thumbnail($size = 'post-thumbnail') {
        if (has_post_thumbnail()) {
            the_post_thumbnail($size ? $size : 'post-thumbnail');
        }
    }
}

// END POSTS FUNCTIONS