<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}
if (!class_exists( 'WooCommerce' )) {
    return;
}

use Elementor\Controls_Manager;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;

/**
 * Elementor TemplateMelaCore_Elementor_Product_Category
 * @since 1.0.0
 */
class TemplateMelaCore_Elementor_Product_Category extends Elementor\Widget_Base {

    public function get_categories() {
        return array('tmcore-addons');
    }

    /**
     * Get widget name.
     *
     * Retrieve tabs widget name.
     *
     * @return string Widget name.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_name() {
        return 'tmcore-product-categories';
    }

    /**
     * Get widget title.
     *
     * Retrieve tabs widget title.
     *
     * @return string Widget title.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_title() {
        return __('TM Product Category', 'templatemela-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve tabs widget icon.
     *
     * @return string Widget icon.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_icon() {
        return 'eicon-image-box';
    }

    /**
     * Register tabs widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since  1.0.0
     * @access protected
     */
    protected function register_controls() {

        //Section Query
        $this->start_controls_section(
            'section_setting',
            [
                'label' => __('Settings', 'templatemela-core'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'categories_name',
            [
                'label' => __('Alternate Name', 'templatemela-core'),
                'type'  => Controls_Manager::TEXT,
            ]
        );

        $this->add_control(
            'categories',
            [
                'label'    => __('Categories', 'templatemela-core'),
                'type'     => Controls_Manager::SELECT2,
                'label_block' => true,
                'options'  => $this->get_product_categories(),
                'multiple' => false,
            ]
        );

        $this->add_control(
            'categories_style',
            [
                'label'        => __('Style', 'templatemela-core'),
                'type'         => Controls_Manager::SELECT,
                'default'      => '1',
                'options'      => [
                    '1' => __('Style 1', 'templatemela-core'),
                    '2' => __('Style 2', 'templatemela-core'),
                    '3' => __('Style 3', 'templatemela-core'),
                    '4' => __('Style 4', 'templatemela-core'),
                    '5' => __('Style 5', 'templatemela-core'),
                    '6' => __('Style 6', 'templatemela-core'),
                ],
                'prefix_class' => 'product-cat-style-',
            ]
        );

        $this->add_control(
            'category_image',
            [
                'label'      => __('Choose Image', 'templatemela-core'),
                'default'    => [
                    'url' => Elementor\Utils::get_placeholder_image_src(),
                ],
                'type'       => Controls_Manager::MEDIA,
                'show_label' => false,
            ]

        );

        $this->add_group_control(
            Elementor\Group_Control_Image_Size::get_type(),
            [
                'name'      => 'image', // Usage: `{name}_size` and `{name}_custom_dimension`, in this case `category_image_size` and `category_image_custom_dimension`.
                'default'   => 'full',
                'separator' => 'none',
            ]
        );

        $this->end_controls_section();

        // WRAPPER STYLE
        $this->start_controls_section(
            'section_style_category_wrapper',
            [
                'label' => __('Wrapper', 'templatemela-core'),
                'tab'   => Controls_Manager::TAB_STYLE,

            ]
        );

        $this->add_responsive_control(
			'category_align',
			[
				'label' => esc_html__( 'Alignment', 'templatemela-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => esc_html__( 'Left', 'templatemela-core' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'templatemela-core' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => esc_html__( 'Right', 'templatemela-core' ),
						'icon' => 'eicon-text-align-right',
					],
				],
				'selectors' => [
					'{{WRAPPER}} .product-cat' => 'text-align: {{VALUE}}',
				],
			]
		);

        $this->add_responsive_control(
            'padding_category_wrapper',
            [
                'label'      => __('Padding', 'templatemela-core'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .product-cat' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'margin_category_wrapper',
            [
                'label'      => __('Margin', 'templatemela-core'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors'  => [
                    '{{WRAPPER}} .product-cat' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'color_category_wrapper',
            [
                'label'     => __('Background Color', 'templatemela-core'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .product-cat' => 'background: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name'        => 'category_wrapper_border',
                'placeholder' => '1px',
                'default'     => '1px',
                'selector'    => '{{WRAPPER}} .product-cat',
                'separator'   => 'before',
            ]
        );

        $this->add_control(
            'category_wrapper_radius',
            [
                'label'      => __('Border Radius', 'templatemela-core'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .product-cat' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'category_wrapper_box_shadow',
                'selector' => '{{WRAPPER}} .product-cat',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'category_image_style',
            [
                'label' => __('Image', 'templatemela-core'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );
		
		$this->add_control(
            'show_image',
            [
                'label' => __('Show Image', 'templatemela-core'),
                'type'  => Controls_Manager::SWITCHER,
                'default'   => 'yes',
            ]
        );

        $this->add_responsive_control(
			'category_image_width',
			[
				'label' => esc_html__( 'Width', 'templatemela-core' ) . ' (%)',
				'type' => Controls_Manager::SLIDER,
                'condition' => [
                    'categories_style' => [
                        '6',
                    ],
				],
				'default' => [
					'size' => 30,
					'unit' => '%',
				],
				'tablet_default' => [
					'unit' => '%',
				],
				'mobile_default' => [
					'unit' => '%',
				],
				'size_units' => [ '%' ],
				'range' => [
					'%' => [
						'min' => 5,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .cat-image' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);

        $this->add_responsive_control(
            'category_image_margin',
            [
                'label'      => __('Margin', 'templatemela-core'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors'  => [
                    '{{WRAPPER}} .product-cat .cat-image' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name'        => 'category_image_border',
                'placeholder' => '1px',
                'default'     => '1px',
                'selector'    => '{{WRAPPER}} .product-cat .cat-image img',
                'separator'   => 'before',
            ]
        );

        $this->add_control(
            'category_image_radius',
            [
                'label'      => __('Image Radius', 'templatemela-core'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .product-cat .cat-image img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();


        $this->start_controls_section(
            'title_style',
            [
                'label' => __('Title', 'templatemela-core'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'tilte_typography',
                'selector' => '{{WRAPPER}} .cat-title',
            ]
        );

        $this->start_controls_tabs('tab_title');
        $this->start_controls_tab(
            'tab_title_normal',
            [
                'label' => __('Normal', 'templatemela-core'),
            ]
        );
        $this->add_control(
            'title_color',
            [
                'label' => __('Color', 'templatemela-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .cat-title a' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'title_background',
            [
                'label' => __('Background', 'templatemela-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .cat-title ' => 'background: {{VALUE}};',
                ],
            ]
        );
        $this->end_controls_tab();
        $this->start_controls_tab(
            'tab_title_hover',
            [
                'label' => __('Hover', 'templatemela-core'),
            ]
        );
        $this->add_control(
            'title_color_hover',
            [
                'label' => __('Hover Color', 'templatemela-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .cat-title:hover a ' => 'color: {{VALUE}};',
                ],
            ]
        );
        $this->add_control(
            'title_background_hover',
            [
                'label' => __('Background Hover', 'templatemela-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .cat-title:hover ' => 'background: {{VALUE}};',
                ],
            ]
        );
        $this->end_controls_tab();
        $this->end_controls_tabs();


        $this->add_responsive_control(
            'title_padding',
            [
                'label'      => __( 'Padding', 'templatemela-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'separator' => 'before',
                'selectors'  => [
                    '{{WRAPPER}} .cat-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );


        $this->end_controls_section();


        $this->start_controls_section(
            'total_style',
            [
                'label' => __('Total', 'templatemela-core'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'show_total',
            [
                'label' => __('Show Total', 'templatemela-core'),
                'default' => 'yes',
                'type'  => Controls_Manager::SWITCHER,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'total_typography',
                'selector' => '{{WRAPPER}} .cat-total',
            ]
        );

        $this->start_controls_tabs('tab_total');
        $this->start_controls_tab(
            'tab_total_normal',
            [
                'label' => __('Normal', 'templatemela-core'),
            ]
        );
        $this->add_control(
            'total_color',
            [
                'label' => __('Color', 'templatemela-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .cat-total' => 'color: {{VALUE}};',
                ],
            ]
        );
        $this->add_control(
            'total_background',
            [
                'label' => __('Background', 'templatemela-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .cat-total ' => 'background: {{VALUE}};',
                ],
            ]
        );
        $this->end_controls_tab();


        $this->start_controls_tab(
            'tab_total_hover',
            [
                'label' => __('Hover', 'templatemela-core'),
            ]
        );
        $this->add_control(
            'total_color_hover',
            [
                'label' => __('Color Hover', 'templatemela-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .product-cat:hover .cat-total' => 'color: {{VALUE}};',
                ],
            ]
        );
        $this->add_control(
            'total_background_hover',
            [
                'label' => __('Background Hover', 'templatemela-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .product-cat:hover .cat-total ' => 'background: {{VALUE}};',
                ],
            ]
        );
        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_responsive_control(
            'total_padding',
            [
                'label'      => __( 'Padding', 'templatemela-core' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', 'em', '%' ],
                'separator' => 'before',
                'selectors'  => [
                    '{{WRAPPER}} .cat-total' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'sub_category_style',
            [
                'label' => __('Sub Categories', 'templatemela-core'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'show_sub_categories',
            [
                'label' => __('Show Sub Categories', 'templatemela-core'),
                'default' => 'yes',
                'type'  => Controls_Manager::SWITCHER,
            ]
        );

        $this->add_control(
            'sub_category_limit',
            [
                'label' => esc_html__('Sub Category Limit', 'templatemela-core'),
                'type' => Controls_Manager::NUMBER,
                'default' => 3,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'sub_category_typography',
                'selector' => '{{WRAPPER}} .sub-categories li a',
            ]
        );

        $this->start_controls_tabs('sub_category');

        $this->start_controls_tab(
            'sub_category_normal',
            [
                'label' => __('Normal', 'templatemela-core'),
            ]
        );

        $this->add_control(
            'sub_category_color',
            [
                'label' => __('Color', 'templatemela-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .sub-categories li a' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'sub_category_hover',
            [
                'label' => __('Hover', 'templatemela-core'),
            ]
        );

        $this->add_control(
            'sub_category_color_hover',
            [
                'label' => __('Color Hover', 'templatemela-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .sub-categories li a:hover' => 'color: {{VALUE}};',
                ],
            ]
        );
        $this->end_controls_tab();

    }


    protected function get_product_categories() {
        $categories = get_terms(array(
                'taxonomy'   => 'product_cat',
                'hide_empty' => false,
            )
        );
        $results    = array();
        if (!is_wp_error($categories)) {
            foreach ($categories as $category) {
                $results[$category->slug] = $category->name;
            }
        }
        return $results;
    }

    /**
     * Render tabs widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since  1.0.0
     * @access protected
     */
    protected function render() {
        $settings = $this->get_settings_for_display();

        if (empty($settings['categories'])) {
            echo esc_html__('Choose Category', 'templatemela-core');
            return;
        }

        $category = get_term_by('slug', $settings['categories'], 'product_cat');
        $count = 0;
        if(!empty($category)){
        $child_args = array(
            'taxonomy' => 'product_cat',
            'hide_empty' => false,
            'parent'   => $category->term_id,
        );
        $child_product_cats = get_terms( $child_args );
        }

        if (!is_wp_error($category) && !empty($category)) {

            if (!empty($settings['category_image']['id'])) {
                $image = Group_Control_Image_Size::get_attachment_image_src($settings['category_image']['id'], 'image', $settings);
            } else {
                $thumbnail_id = get_term_meta($category->term_id, 'thumbnail_id', true);
                if (!empty($thumbnail_id)) {
                    $image = wp_get_attachment_url($thumbnail_id);
                } else {
                    $image = wc_placeholder_img_src();
                }
            }
            
            ?>

            <div class="product-cat">
                
                <?php if ($settings['categories_style'] == '1'): ?>
                <div class="cat-image">
					<?php if ($settings['show_image']) : ?>
                    <a class="link_category_product" href="<?php echo esc_url(get_term_link($category)); ?>" title="<?php echo esc_attr($category->name); ?>">
                        <img src="<?php echo esc_url_raw($image); ?>" alt="<?php echo esc_html($category->name); ?>">
                    </a>
					<?php endif; ?>
                    <?php if ($settings['show_total']): ?>
                        <div class="cat-total"><?php echo esc_html($category->count) . ' ' . esc_html__('Products', 'templatemela-core'); ?></div>
                    <?php endif; ?>
                </div>
                
                <div class="cat-title">
                    <a href="<?php echo esc_url(get_term_link($category)); ?>" title="<?php echo esc_attr($category->name); ?>">
                        <?php echo empty($settings['categories_name']) ? esc_html($category->name) : $settings['categories_name']; ?>
                    </a>
                </div>

                <?php if ($settings['show_sub_categories']): ?>
                    <ul class="sub-categories">
                        <?php foreach ( $child_product_cats as $child_product_cat ) { ?>
                            <li><a href=" <?php echo esc_url(get_term_link($child_product_cat)); ?>"><?php echo $child_product_cat->name ?></a></li>
                            <?php $count++; 
                            if ($count == $settings['sub_category_limit']){
                                break;
                            }
                        }
                        ?>
                    </ul>
                <?php endif; ?>

                <?php elseif ($settings['categories_style'] == '2' || $settings['categories_style'] == '4' || $settings['categories_style'] == '5'): ?>
				    <?php if ($settings['show_image']) : ?>
                    <div class="cat-image">
                        <a class="link_category_product" href="<?php echo esc_url(get_term_link($category)); ?>" title="<?php echo esc_attr($category->name); ?>">
                            <img src="<?php echo esc_url_raw($image); ?>" alt="<?php echo esc_html($category->name); ?>">
                        </a>
					</div>
				    <?php endif; ?>
                    <div class="product-cat-caption">
                        <div class="cat-title">
                            <a href="<?php echo esc_url(get_term_link($category)); ?>" title="<?php echo esc_attr($category->name); ?>">
                            <?php echo empty($settings['categories_name']) ? esc_html($category->name) : wp_kses_post($settings['categories_name']); ?>
                            </a>
                        </div>
                        <?php if ($settings['show_total']): ?>
                            <div class="cat-total"><?php echo esc_html($category->count) . ' ' . esc_html__('Products', 'templatemela-core'); ?></div>
                        <?php endif; ?>
                    </div>

                <?php elseif ($settings['categories_style'] == '6'): ?>
                    <div class="cat-image">
                        <?php if ($settings['show_image']) : ?>
                        <a class="link_category_product" href="<?php echo esc_url(get_term_link($category)); ?>" title="<?php echo esc_attr($category->name); ?>">
                            <img src="<?php echo esc_url_raw($image); ?>" alt="<?php echo esc_html($category->name); ?>">
                        </a>
                        <?php endif; ?>
                        <?php if ($settings['show_total']): ?>
                            <div class="cat-total"><?php echo esc_html($category->count) . ' ' . esc_html__('Products', 'templatemela-core'); ?></div>
                        <?php endif; ?>
                    </div>

                    <div class="cat-contents">
                        <div class="cat-title">
                            <a href="<?php echo esc_url(get_term_link($category)); ?>" title="<?php echo esc_attr($category->name); ?>">
                                <?php echo empty($settings['categories_name']) ? esc_html($category->name) : $settings['categories_name']; ?>
                            </a>
                        </div>

                        <?php if ($settings['show_sub_categories']): ?>
                            <ul class="sub-categories">
                                <?php foreach ( $child_product_cats as $child_product_cat ) { ?>
                                    <li><a href=" <?php echo esc_url(get_term_link($child_product_cat)); ?>"><?php echo $child_product_cat->name ?></a></li>
                                    <?php $count++; 
                                    if ($count == $settings['sub_category_limit']){
                                        break;
                                    }
                                }
                                ?>
                            </ul>
                        <?php endif; ?>
                    </div>

                <?php else: ?>
                    <div class="cat-image">
						<?php if ($settings['show_image']) : ?>
                        <a class="link_category_product" href="<?php echo esc_url(get_term_link($category)); ?>" title="<?php echo esc_attr($category->name); ?>">
                            <img src="<?php echo esc_url_raw($image); ?>" alt="<?php echo esc_html($category->name); ?>">
                        </a>
						<?php endif; ?>
                        <?php if ($settings['show_total']): ?>
                            <div class="cat-total"><?php echo esc_html($category->count) . ' ' . esc_html__('Products', 'templatemela-core'); ?></div>
                        <?php endif; ?>
                    </div>
                    <div class="cat-title">
                        <a href="<?php echo esc_url(get_term_link($category)); ?>" title="<?php echo esc_attr($category->name); ?>">
                            <?php echo empty($settings['categories_name']) ? esc_html($category->name) : $settings['categories_name']; ?>
                        </a>
                    </div>
                <?php endif; ?>
            </div>
            <?php

        }

    }
}

$widgets_manager->register(new TemplateMelaCore_Elementor_Product_Category());

